﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion
{
    using System;
    using System.IO;
    using System.Xml;
    using System.Xml.Serialization;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// 依存関係管理クラス
    /// </summary>
    public partial class DependencyManager : IXmlSerializable
    {
        /// <summary>
        /// 出力情報クラス
        /// </summary>
        private class DependentOutputItem : IDependentOutputItem
        {
            #region ** 固定値

            // 要素名
            public const string ElementName = "OutputInfo";

            // 属性名
            private const string KeyAttributeName = "Key";
            private const string NameAttributeName = "Name";

            #endregion

            private string key = string.Empty;
            private string name = string.Empty;
            private Lazy<string> fullPath;

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="owner">DependencyOutputItemInfoを所有するDependencyManager</param>
            /// <param name="name">出力情報名</param>
            public DependentOutputItem(string key, string name)
            {
                if (null == key) { throw new ArgumentNullException("key"); }
                if (null == name) { throw new ArgumentNullException("name"); }
                if (0 == name.Length) { throw new ArgumentException("name"); }

                this.key = key;
                this.name = name.ToString();

                // Path.GetFullPath() を毎回評価するとパフォーマンスに影響が出るので、キャッシュしています。
                this.fullPath = new Lazy<string>(() =>
                {
                    // 絶対パスの場合はそのまま返す
                    if (Path.IsPathRooted(this.FilePath)) { return FilePath; }
                    if (null == this.Owner) { return string.Empty; }
                    return Path.Combine(this.Owner.BaseDirectoryAbsolutePath, this.FilePath).GetFullPath();
                });
            }

            /// <summary>
            /// DependencyManagerを取得します。
            /// </summary>
            public DependencyManager Owner { get; set; }

            /// <summary>
            /// キーを取得します。
            /// </summary>
            public string Key
            {
                get { return this.key; }
            }

            /// <summary>
            /// 出力情報名を取得または設定します。
            /// </summary>
            public string Name
            {
                get { return this.name; }
            }

            /// <summary>
            /// 出力ファイルのパスを取得します。
            /// </summary>
            public string FilePath
            {
                get { return this.name; }
            }

            /// <summary>
            /// 出力ファイルの絶対パスを取得します。
            /// </summary>
            public string AbsoluteFilePath => this.fullPath.Value;

            /// <summary>
            /// 最終更新時間を取得します。
            /// </summary>
            public DateTime LastUpdatedTime
            {
                get
                {
                    if (!File.Exists(this.AbsoluteFilePath))
                    {
                        throw new DependencyFileNotFoundException(this.AbsoluteFilePath);
                    }

                    return File.GetLastWriteTime(this.AbsoluteFilePath);
                }
            }

            //-----------------------------------------------------------------
            // XML入出力
            //-----------------------------------------------------------------

            /// <summary>
            /// XML要素から出力情報を読み込みます。
            /// </summary>
            /// <param name="owner">DependencyOutputItemInfoを所有するDependencyManager</param>
            /// <param name="xmlElement">XML要素</param>
            /// <returns></returns>
            public static DependentOutputItem FromXml(XmlElement xmlElement)
            {
                if (null == xmlElement) { throw new ArgumentNullException("xmlElement"); }

                if (ElementName != xmlElement.Name) { throw new DependencyManagerInvalidFileFormatException(); }
                if (!xmlElement.HasAttribute(KeyAttributeName)) { throw new DependencyManagerInvalidFileFormatException(); }
                if (!xmlElement.HasAttribute(NameAttributeName)) { throw new DependencyManagerInvalidFileFormatException(); }

                return new DependentOutputItem(
                    xmlElement.GetAttribute(KeyAttributeName),
                    xmlElement.GetAttribute(NameAttributeName));
            }

            /// <summary>
            /// 出力情報をXML出力します。
            /// </summary>
            /// <param name="document">XMLドキュメント</param>
            /// <returns>XML要素</returns>
            public XmlElement ToXml(XmlDocument document)
            {
                Assertion.Argument.NotNull(document);

                // OutuptInfo
                XmlElement outputInfoElement = document.CreateElement(ElementName);
                outputInfoElement.SetAttribute(KeyAttributeName, this.key);
                outputInfoElement.SetAttribute(NameAttributeName, this.name);

                return outputInfoElement;
            }

            //-----------------------------------------------------------------
            // 出力ファイルの削除
            //-----------------------------------------------------------------

            /// <summary>
            /// 出力ファイルを削除します。
            /// </summary>
            public void DeleteFile()
            {
                try
                {
                    if (File.Exists(this.AbsoluteFilePath))
                    {
                        File.Delete(this.AbsoluteFilePath);
                    }
                }
                catch
                {
                }
            }
        }
    }
}
