﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion
{
    using System;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.Linq;
    using ToolDevelopmentKit;
    using ToolDevelopmentKit.Collections;

    /// <summary>
    /// 処理の実行状態を管理する基本クラスです。
    /// </summary>
    /// <typeparam name="TContext">コンテキストの型を指定します。</typeparam>
    public abstract class ConversionProcessor<TContext> : IConversionProcessor
        where TContext : ConversionContext
    {
        private readonly object stateLock = new object();

        private ConversionProcessState state = ConversionProcessState.Ready;
        private ObservableList<IConversionProcessor> dependedProcessors = new ObservableList<IConversionProcessor>();

        // すでにイベント登録済みかどうかを高速に判断するために ObservableList とは別に HashSet を用意している
        private HashSet<IConversionProcessor> dependedProcessorsHashSet = new HashSet<IConversionProcessor>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        protected ConversionProcessor()
        {
            this.IsForced = false;
            this.IsSkiped = true;
            this.dependedProcessors.CollectionChanged += OnDependenciesChanged;
        }

        /// <summary>
        /// IConversionProcessor の状態が変更されると発生します。
        /// </summary>
        public event EventHandler<ConversionProcessStateEventArgs> StateChanged;

        /// <summary>
        /// 依存する IConversionProcessor のリストを取得します。
        /// </summary>
        public IList<IConversionProcessor> Dependencies
        {
            get { return this.dependedProcessors; }
        }

        /// <summary>
        /// IConversionProcessor の状態を取得します。
        /// </summary>
        public ConversionProcessState State
        {
            get { return this.state; }
        }

        /// <summary>
        /// 強制的に実行するかどうかを取得または設定します。
        /// </summary>
        public bool IsForced { get; set; }

        /// <summary>
        /// 処理をスキップしたかどうかを取得または設定します。
        /// </summary>
        public bool IsSkiped { get; set; }

        /// <summary>
        /// 処理を実行します。
        /// </summary>
        /// <param name="context">コンテキストを指定します。</param>
        public void Run(ConversionContext context)
        {
            Ensure.Argument.NotNull(context);
            Ensure.Argument.True(context is TContext);

            switch (this.state)
            {
                case ConversionProcessState.NotReady:
                    throw new Exception("internal error : processor is not prepared.");

                case ConversionProcessState.Done:
                    return;
            }

            this.IsSkiped = true;

            try
            {
                TContext contextWork = context as TContext;
                bool result = true;

                if (this.PreRunInternal(contextWork) || this.IsForced)
                {
                    this.SetState(ConversionProcessState.Running);

                    result = this.RunInternal(contextWork);
                    this.IsSkiped = false;
                }

                if (result)
                {
                    this.PostRunInternal(contextWork);
                    this.SetState(ConversionProcessState.Done);
                }
                else
                {
                    this.SetState(ConversionProcessState.Error);
                }
            }
            catch
            {
                this.SetState(ConversionProcessState.Error);
                throw;
            }
        }

        protected virtual bool PreRunInternal(TContext context)
        {
            return true;
        }

        protected abstract bool RunInternal(TContext context);

        protected virtual void PostRunInternal(TContext context)
        {
        }

        protected virtual void OnStateChanged(ConversionProcessStateEventArgs e)
        {
            Assertion.Argument.NotNull(e);

            if (this.StateChanged != null)
            {
                this.StateChanged(this, e);
            }
        }

        private void SetState(ConversionProcessState state)
        {
            lock (this.stateLock)
            {
                if (this.state == state) { return; }


                this.state = state;

                this.OnStateChanged(new ConversionProcessStateEventArgs(state));
            }
        }

        private void UpdateState()
        {
            switch (this.state)
            {
                case ConversionProcessState.NotReady:
                    if (this.IsPrepared())
                    {
                        this.SetState(ConversionProcessState.Ready);
                    }
                    break;

                case ConversionProcessState.Ready:
                    if (!this.IsPrepared())
                    {
                        this.SetState(ConversionProcessState.NotReady);
                    }
                    break;
            }
        }

        private bool IsPrepared()
        {
            foreach (var processor in this.dependedProcessors)
            {
                if (processor.State != ConversionProcessState.Done) { return false; }
            }

            return true;
        }

        private void AddProcessorStateChanged(IEnumerable<IConversionProcessor> processors)
        {
            Assertion.Argument.NotNull(processors);

            foreach (var processor in processors)
            {
                if (this.dependedProcessorsHashSet.Contains(processor))
                {
                    continue;
                }

                this.dependedProcessorsHashSet.Add(processor);
                processor.StateChanged += OnDependedProcessorStateChanged;
            }
        }

        private void RemoveProcessorStateChanged(IEnumerable<IConversionProcessor> processors)
        {
            Assertion.Argument.NotNull(processors);

            foreach (var processor in processors)
            {
                processor.StateChanged -= OnDependedProcessorStateChanged;
            }
        }

        private void OnDependedProcessorStateChanged(object sender, ConversionProcessStateEventArgs e)
        {
            Assertion.Argument.NotNull(e);

            UpdateState();
        }

        private void OnDependenciesChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            Assertion.Argument.NotNull(e);

            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    this.AddProcessorStateChanged(e.NewItems.Cast<IConversionProcessor>());
                    break;

                case NotifyCollectionChangedAction.Remove:
                    this.RemoveProcessorStateChanged(e.OldItems.Cast<IConversionProcessor>());
                    break;

                case NotifyCollectionChangedAction.Replace:
                    this.AddProcessorStateChanged(e.NewItems.Cast<IConversionProcessor>());
                    this.RemoveProcessorStateChanged(e.OldItems.Cast<IConversionProcessor>());
                    break;

                case NotifyCollectionChangedAction.Reset:
                    throw new Exception("internal error");
            }

            UpdateState();
        }
    }
}
