﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion
{
    using System;
    using System.Diagnostics;
    using NintendoWare.SoundFoundation.Logs;
    using ToolDevelopmentKit;

    /// <summary>
    /// コンソールベースのコンバート機能を提供する基本クラスです。
    /// </summary>
    public abstract class ConsoleBasedConverter<TContext> : IConsoleBasedConverter<TContext>
        where TContext : ConversionContext
    {
        private TContext context;

        public string LogPrefix { get; set; }

        protected TimeSpan? Timeout { get; set; }

        public bool Run(TContext context)
        {
            bool result = false;

            Process process = new Process();
            this.InitializeProcess(context, process);

            try
            {
                this.context = context;

                this.AddLine(
                    $"{LogPrefix ?? string.Empty}{process.StartInfo.FileName} {process.StartInfo.Arguments}",
                    this.CreateInformationLine);

                if (!process.Start())
                {
                    throw new Exception("internal error : failed to start convert process.");
                }

                process.BeginOutputReadLine();
                process.BeginErrorReadLine();

                if (this.Timeout.HasValue)
                {
                    if (!process.WaitForExit((int)this.Timeout.Value.TotalMilliseconds))
                    {
                        process.Kill();
                        throw new TimeoutException();
                    }
                }
                else
                {
                    process.WaitForExit();
                }

                result = process.ExitCode == 0;
            }
            finally
            {
                this.PostRun(context, process);
                process.Close();
                this.context = null;
            }

            return result;
        }

        protected virtual void InitializeProcess(TContext context, Process process)
        {
            Assertion.Argument.NotNull(process);

            process.StartInfo.FileName = this.GetFileName(context);
            process.StartInfo.Arguments = this.GetArguments(context);
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardOutput = true;
            process.StartInfo.RedirectStandardError = true;

            process.OutputDataReceived += OnOutputDataReceived;
            process.ErrorDataReceived += OnErrorDataReceived;
        }

        protected virtual void PostRun(TContext context, Process process)
        {
        }

        protected abstract string GetFileName(TContext context);

        protected abstract string GetArguments(TContext context);

        protected virtual InformationLine CreateInformationLine(string message)
        {
            return new InformationLine(message);
        }

        protected virtual ErrorLine CreateErrorLine(string message)
        {
            return new ErrorLine(message);
        }

        private void OnOutputDataReceived(object sender, DataReceivedEventArgs e)
        {
            if (e.Data != null && e.Data.Length > 0)
            {
                this.AddLine(e.Data, this.CreateInformationLine);
            }
        }

        private void OnErrorDataReceived(object sender, DataReceivedEventArgs e)
        {
            if (e.Data != null && e.Data.Length > 0)
            {
                this.AddLine(e.Data, this.CreateErrorLine);
            }
        }

        private void AddLine(string message, Func<string, OutputLine> createLine)
        {
            var line = createLine(message);

            if (line != null)
            {
                this.context?.Logger?.AddLine(line);
            }
        }
    }
}
