﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;

namespace NintendoWare.SoundFoundation.Commands
{
    /// <summary>
    /// コマンドの状態を表します。
    /// </summary>
    [Flags]
    public enum CommandStatus
    {
        /// <summary>
        /// コマンドがサポートされていないことを示します。
        /// </summary>
        Unsupported = 0x00,

        /// <summary>
        /// コマンドがサポートされていることを示します。
        /// </summary>
        Supported = 0x01,

        /// <summary>
        /// コマンドが有効であることを示します。
        /// </summary>
        Enabled = 0x02,

        /// <summary>
        /// コマンドが表示されていることを示します。
        /// </summary>
        Visible = 0x04,

        /// <summary>
        /// コマンドがチェックされていることを示します。
        /// </summary>
        Checked = 0x08,

        /// <summary>
        /// コマンドがサポートされている、かつ有効であることを示します。
        /// </summary>
        SupportedAndEnabled = Supported | Enabled,

        /// <summary>
        /// コマンドがサポートされている、かつ表示されていることを示します。
        /// </summary>
        SupportedAndVisible = Supported | Visible,

        /// <summary>
        /// コマンドがサポートされている、かつ有効、表示されていることを示します。
        /// </summary>
        SupportedAndEnabledAndVisible = Supported | Enabled | Visible,

        /// <summary>
        /// コマンドがサポートされている、かつチェックされていることを示します。
        /// </summary>
        SupportedAndChecked = Supported | Checked,
    }

    /// <summary>
    /// CommandStatus の拡張機能を提供します。
    /// </summary>
    public static class CommandStatusEx
    {
        /// <summary>
        /// コマンドの状態が有効かどうか調べます。
        /// </summary>
        /// <param name="value">コマンド。</param>
        /// <returns>有効なら true、無効なら false。</returns>
        public static bool IsSupported(this CommandStatus value)
        {
            return ((value & CommandStatus.Supported) != 0);
        }

        /// <summary>
        /// コマンドの状態が有効かどうか調べます。
        /// </summary>
        /// <param name="value">コマンド。</param>
        /// <returns>有効なら true、無効なら false。</returns>
        public static bool IsEnabled(this CommandStatus value)
        {
            if (!value.IsSupported()) { return false; }
            return ((value & CommandStatus.Enabled) != 0);
        }

        /// <summary>
        /// コマンドの表示状態を調べます。
        /// </summary>
        /// <param name="value">コマンド。</param>
        /// <returns>表示状態なら true、非表示状態なら false。</returns>
        public static bool IsVisible(this CommandStatus value)
        {
            if (!value.IsSupported()) { return false; }
            return ((value & CommandStatus.Visible) != 0);
        }

        /// <summary>
        /// コマンドのチェック状態を調べます。
        /// </summary>
        /// <param name="value">コマンド。</param>
        /// <returns>チェックありなら true、チェックなしなら false。</returns>
        public static bool IsChecked(this CommandStatus value)
        {
            if (!value.IsSupported()) { return false; }
            return ((value & CommandStatus.Checked) != 0);
        }
    }
}
