﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Codecs
{
    using System.IO;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// サンプルを格納するパケットです。
    /// </summary>
    public class Packet
    {
        private const int InvalidSampleCount = -1;

        private byte[] samples;
        private int validSampleCount = InvalidSampleCount;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="samples">パケットに格納するサンプルを指定します。</param>
        public Packet(byte[] samples)
        {
            Ensure.Argument.NotNull(samples);
            this.samples = samples;
        }

        /// <summary>
        /// 格納されているサンプルを取得します。
        /// </summary>
        public byte[] Samples
        {
            get { return this.samples; }
        }

        /// <summary>
        /// 格納されているサンプルの有効な長さを取得または設定します。
        /// </summary>
        public int ValidSamplesLength
        {
            get
            {
                return this.validSampleCount == InvalidSampleCount ?
                    this.samples.Length : this.validSampleCount;
            }

            set
            {
                Ensure.Argument.True(value >= 0);
                Ensure.Argument.True(value <= this.samples.Length);

                this.validSampleCount = value;
            }
        }

        /// <summary>
        /// ストリーム終端かどうかを調べます。
        /// </summary>
        public bool IsEndOfStream
        {
            get { return this.samples.Length == 0; }
        }

        /// <summary>
        /// パケットを生成します。
        /// </summary>
        /// <param name="samples">パケットに格納するサンプルを指定します。</param>
        /// <returns>生成したパケットを返します。</returns>
        public static Packet Create(byte[] samples)
        {
            return new Packet(samples);
        }

        /// <summary>
        /// ストリーム終端パケットを生成します。
        /// </summary>
        /// <returns>生成したストリーム終端パケットを返します。</returns>
        public static Packet CreateEndOfStreamPacket()
        {
            return new Packet(new byte[0]);
        }

        /// <summary>
        /// 格納されているサンプルを参照するストリームを作成します。
        /// </summary>
        /// <returns></returns>
        public Stream CreateStream()
        {
            return new MemoryStream(this.samples, false);
        }
    }
}
