﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Binarization
{
    using System;
    using System.Diagnostics;
    using Core;
    using ToolDevelopmentKit;

    internal class DomObjectReferenceWriter : IDomObjectProcessor
    {
        private string originTag = string.Empty;
        private string originPath = string.Empty;

        public string OriginTag
        {
            get { return this.originTag; }
            set
            {
                Ensure.Argument.NotNull(value);
                this.originTag = value;
            }
        }

        public string OriginPath
        {
            get { return this.originPath; }
            set
            {
                Ensure.Argument.NotNull(value);
                this.originPath = value;
            }
        }

        public bool HasType { get; set; }

        public bool HasOffset { get; set; }

        public bool HasSize { get; set; }

        public SizeType SizeType { get; set; }

        public Type OffsetValueType { get; set; }

        public Type SizeValueType { get; set; }

        public void Run(DomObject domObject, IDomContext context)
        {
            if (!HasOffset && !HasSize)
            {
                throw new ArgumentException("Offset or Size must be Included.");
            }

            DomWriterContext writerContext = context as DomWriterContext;
            if (null == writerContext)
            {
                throw new ArgumentException("objectProcessor must be DomWriterContext.");
            }

            if (null == writerContext.CurrentElement)
            {
                throw new ArgumentException("internal error.");
            }

            DomObject originDomObject = this.GetOrigin(writerContext);

            if (this.HasType)
            {
                writerContext.Writer.Write(GetElementType(domObject));   // データタイプ 2byte
                writerContext.Writer.Write((UInt16)0);                   // パディング   2byte
            }

            if (this.HasOffset)
            {
                // オフセット ?byte（プラットフォーム依存）
                AddressWriter addressCommiter = this.GetAddressWriter(
                    domObject.Value,
                    (originDomObject == null) ? null : originDomObject.Value,
                    writerContext.ReferenceResolver);

                addressCommiter.Reserve(writerContext);
                writerContext.ReferenceResolver.RegisterAddressCommiter(addressCommiter);
            }

            if (this.HasSize)
            {
                // サイズ ?byte（プラットフォーム依存）
                AddressWriter sizeCommiter =
                    this.GetSizeWriter(domObject.Value, writerContext.ReferenceResolver);

                sizeCommiter.Reserve(writerContext);
                writerContext.ReferenceResolver.RegisterAddressCommiter(sizeCommiter);
            }
        }

        private DomObject GetOrigin(DomWriterContext context)
        {
            Assertion.Argument.NotNull(context);

            DomObject taggedDomObject = this.GetOriginFromTag(context);
            return this.GetOriginFromPath(
                (taggedDomObject == null) ? context.CurrentElement : taggedDomObject);
        }

        private DomObject GetOriginFromTag(DomWriterContext context)
        {
            Assertion.Argument.NotNull(context);

            if (this.originTag.Length == 0)
            {
                return null;
            }

            return context.ReferenceResolver.GetDomObject(this.originTag);
        }

        private DomObject GetOriginFromPath(DomObject target)
        {
            Assertion.Argument.NotNull(target);

            if (this.originPath.Length == 0)
            {
                return target;
            }

            return new TreeTracer('/').Trace(target, this.originPath) as DomObject;
        }

        private UInt16 GetElementType(DomObject domObject)
        {
            Assertion.Argument.NotNull(domObject);

            if (domObject.Value == null) { return 0; }

            object[] attrs = domObject.Value.GetType().GetCustomAttributes(typeof(DomElementTypeAttribute), false);
            if (attrs.Length == 0) { return 0; }

            return (attrs[0] as DomElementTypeAttribute).Value;
        }

        private AddressWriter GetAddressWriter(object value, object originValue, ObjectReferenceResolver resolver)
        {
            Debug.Assert(null != resolver, "invalid argument");

            return new DomObjectAddressWriter()
            {
                Reference = (null == value) ?
                                   null : resolver.GetObjectReference(value),
                OriginReference = (null == originValue) ?
                                          null : resolver.GetObjectReference(originValue),
                OffsetValueType = this.OffsetValueType,
            };
        }

        private AddressWriter GetSizeWriter(object value, ObjectReferenceResolver resolver)
        {
            Debug.Assert(null != resolver, "invalid argument");

            return new DomObjectSizeWriter()
            {
                Reference = (null == value) ?
                                   null : resolver.GetObjectReference(value),
                SizeType = this.SizeType,
                SizeValueType = this.SizeValueType,
            };
        }
    }
}
