﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Binarization
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using ToolDevelopmentKit;

    /// <summary>
    /// DOMツリーの各要素に対して処理を実行します。
    /// </summary>
    public class DomProcessor
    {
        private DomContext context;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="context">DOMコンテキストを指定します。</param>
        public DomProcessor(DomContext context)
        {
            if (null == context) { throw new ArgumentNullException("context"); }
            this.context = context;
        }

        /// <summary>
        /// DOMコンテキストを取得します。
        /// </summary>
        public IDomContext Context
        {
            get { return this.context; }
        }

        /// <summary>
        /// DOMツリーの各要素に対して、指定された処理を実行します。
        /// </summary>
        /// <param name="objectProcessor">実行するDOMオブジェクトプロセッサを指定します。</param>
        /// <param name="element">DOMツリーの要素を指定します。</param>
        public void Run(IDomObjectProcessor objectProcessor, DomElement element)
        {
            Ensure.Argument.NotNull(objectProcessor);
            Ensure.Argument.NotNull(element);

            RunInternal(objectProcessor, element);
        }

        protected virtual void PreProcess(
            IDomObjectProcessor objectProcessor, DomObject domObject, DomAttributeTarget target)
        {
            Assertion.Argument.NotNull(objectProcessor);
            Assertion.Argument.NotNull(domObject);

            foreach (IDomObjectProcessor preProcessor in
                this.GetCustomProcessors<DomPreProcessorAttribute>(
                objectProcessor, domObject.GetAttributes<DomPreProcessorAttribute>(), target))
            {
                preProcessor.Run(domObject, context);
            }
        }

        protected virtual void PostProcess(
            IDomObjectProcessor objectProcessor, DomObject domObject, DomAttributeTarget target)
        {
            Assertion.Argument.NotNull(objectProcessor);
            Assertion.Argument.NotNull(domObject);

            foreach (IDomObjectProcessor postProcessor in
                this.GetCustomProcessors<DomPostProcessorAttribute>(
                objectProcessor, domObject.GetAttributes<DomPostProcessorAttribute>(), target))
            {
                postProcessor.Run(domObject, context);
            }
        }

        protected virtual void ProcessElement(IDomObjectProcessor objectProcessor, DomElement element)
        {
            Assertion.Argument.NotNull(objectProcessor);
            Assertion.Argument.NotNull(element);
            Assertion.Argument.NotNull(element.Value);

            foreach (DomElementField field in element.Children)
            {
                this.PreProcess(objectProcessor, field, DomAttributeTarget.FieldValues);
                this.ProcessingObject(field);

                bool isDefaultProcess = this.ProcessField(
                    objectProcessor,
                    field,
                    field.IsDepended ? DomAttributeTarget.Field : DomAttributeTarget.FieldValues);

                if (isDefaultProcess && field.IsDepended)
                {
                    if (null == field.Value)
                    {
                        throw new Exception("internal error");
                    }

                    // DomElement の場合は、内部のオブジェクトを処理します。
                    this.RunInternal(objectProcessor, field.DependedObject);
                }

                this.ProcessedObject();
                this.PostProcess(objectProcessor, field, DomAttributeTarget.FieldValues);
            }
        }

        protected virtual bool ProcessField(
            IDomObjectProcessor objectProcessor, DomObject domObject, DomAttributeTarget target)
        {
            Debug.Assert(null != objectProcessor, "invalid argument");
            Debug.Assert(null != domObject, "invalid argument");

            IDomObjectProcessor customProcessor =
                this.GetCustomProcessor<DomCustomProcessorAttribute>(
                objectProcessor, domObject.GetAttributes<DomCustomProcessorAttribute>(), target);

            if (null != customProcessor)
            {
                customProcessor.Run(domObject, context);
                return false;
            }

            if (domObject.IsDepended) { return true; }

            objectProcessor.Run(domObject, this.context);
            return true;
        }

        private void RunInternal(IDomObjectProcessor objectProcessor, DomElement element)
        {
            Assertion.Argument.NotNull(objectProcessor);
            Assertion.Argument.NotNull(element);

            if (null == element.Value)
            {
                throw new ArgumentException("invalid DomElement. DomElement.Object must not be null.");
            }

            this.PreProcess(objectProcessor, element, DomAttributeTarget.Field);
            this.ProcessingObject(element);

            this.ProcessElement(objectProcessor, element);

            this.ProcessedObject();
            this.PostProcess(objectProcessor, element, DomAttributeTarget.Field);
        }

        private void ProcessingObject(DomObject domObject)
        {
            this.context.Push(domObject);
        }

        private void ProcessedObject()
        {
            this.context.Pop();
        }

        private IDomObjectProcessor GetCustomProcessor<TAttribute>(
            IDomObjectProcessor objectProcessor,
            IEnumerable<TAttribute> customProcessors,
            DomAttributeTarget target)
            where TAttribute : DomProcessorAttribute
        {
            Assertion.Argument.NotNull(objectProcessor);
            Assertion.Argument.NotNull(customProcessors);

            Type objectProcessorType = objectProcessor.GetType();

            foreach (DomProcessorAttribute attr in customProcessors)
            {
                if ((attr.Target & target) == DomAttributeTarget.None) { continue; }
                if (attr.ProcessorType != objectProcessorType) { continue; }
                return attr.CustomProcessor;
            }

            return null;
        }

        private IEnumerable<IDomObjectProcessor> GetCustomProcessors<TAttribute>(
            IDomObjectProcessor objectProcessor,
            IEnumerable<TAttribute> customProcessors,
            DomAttributeTarget target)
            where TAttribute : DomProcessorAttribute
        {
            Assertion.Argument.NotNull(objectProcessor);
            Assertion.Argument.NotNull(customProcessors);

            Type objectProcessorType = objectProcessor.GetType();

            foreach (DomProcessorAttribute attr in customProcessors)
            {
                if ((attr.Target & target) == DomAttributeTarget.None) { continue; }
                if (attr.ProcessorType != objectProcessorType) { continue; }
                yield return attr.CustomProcessor;
            }
        }
    }
}
