﻿namespace Opal.Windows.Data
{
    using System;
    using System.Collections.Generic;
    using System.Globalization;
    using System.Windows;
    using System.Windows.Data;
    using System.Windows.Markup;

    /// <summary>
    /// EnumViewEntry から Enum へ変換します。
    /// </summary>
    [ValueConversion(typeof(EnumViewEntry), typeof(Enum))]
    [ContentProperty("ViewEntries")]
    public class EnumViewer : IValueConverter
    {
        private List<EnumViewEntry> viewEntries;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public EnumViewer()
        {
        }

        /// <summary>
        /// 表示登録を取得します。
        /// コンテントプロパティとして設定するために List にしています。
        /// </summary>
        public List<EnumViewEntry> ViewEntries
        {
            get
            {
                if (this.viewEntries == null)
                {
                    this.viewEntries = new List<EnumViewEntry>();
                }

                return this.viewEntries;
            }
        }

        /// <summary>値を変換します。</summary>
        /// <param name="value">バインディング ソースによって生成された値です。</param>
        /// <param name="targetType">バインディング ターゲット プロパティの型です。</param>
        /// <param name="parameter">使用するコンバータ パラメータです。</param>
        /// <param name="culture">コンバータで使用するカルチャです。</param>
        /// <returns>変換された値です。メソッドが null を返す場合は、有効な null 値が使用されています。</returns>
        public object Convert(object value, Type targetType, object parameter, CultureInfo culture)
        {
            Type type = parameter as Type;
            if (type == null || value == null)
            {
                return DependencyProperty.UnsetValue;
            }

            if (Enum.IsDefined(type, value) == false)
            {
                return DependencyProperty.UnsetValue;
            }

            EnumViewEntry result = this.FindEntry(type, value);
            if (result != null)
            {
                return result;
            }

            return DependencyProperty.UnsetValue;
        }

        /// <summary>値を変換します。</summary>
        /// <param name="value">バインディング ターゲットによって生成される値です。</param>
        /// <param name="targetType">変換後の型です。</param>
        /// <param name="parameter">使用するコンバータ パラメータです。</param>
        /// <param name="culture">コンバータで使用するカルチャです。</param>
        /// <returns>変換された値です。メソッドが null を返す場合は、有効な null 値が使用されています。</returns>
        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo culture)
        {
            Type type = parameter as Type;
            if (type == null)
            {
                return DependencyProperty.UnsetValue;
            }

            EnumViewEntry entry = value as EnumViewEntry;
            if (entry != null)
            {
                return Enum.Parse(type, entry.EnumValue);
            }

            return DependencyProperty.UnsetValue;
        }

        /// <summary>
        /// ビュー登録を検索します。
        /// </summary>
        /// <param name="enumType">Enumのタイプです。</param>
        /// <param name="enumValue">Enumの値です。</param>
        /// <returns>検索対象が存在すれば、インスタンスを返します。存在しない場合は、nullを返します。</returns>
        private EnumViewEntry FindEntry(Type enumType, object enumValue)
        {
            if (this.viewEntries != null &&
                this.viewEntries.Count > 0)
            {
                EnumViewEntry foundEntry =
                    this.viewEntries.Find(
                    delegate(EnumViewEntry entry)
                    {
                        object findValue = Enum.Parse(enumType, entry.EnumValue);
                        return enumValue.Equals(findValue);
                    });

                if (foundEntry != null)
                {
                    return foundEntry;
                }
            }

            return null;
        }
    }
}
