﻿using System.Collections.Concurrent;

namespace Opal.Utilities
{
    using System;
    using System.Collections.Generic;
    using Nintendo.ToolFoundation.Contracts;
    using System.IO;
    using System.Linq;
    using System.Runtime.Serialization;
    using System.Xml;
    using System.Xml.Linq;
    using System.Xml.Serialization;
    using System.Xml.XPath;

    /// <summary>
    /// 中間ファイルの XML 操作用ユーティリティです。
    /// </summary>
    public class IfDomUtility
    {
        private static readonly HashSet<string> LoadedAssemblies = new HashSet<string>();

        private static readonly ConcurrentDictionary<string, Type> ElementNameToTypeDict = new ConcurrentDictionary<string, Type>();

        private static readonly ConcurrentDictionary<string, XmlSerializer> XmlSrializerPool = new ConcurrentDictionary<string, XmlSerializer>();

        /// <summary>
        /// T を XmlElement に変換します。
        /// </summary>
        /// <typeparam name="T">変換するオブジェクトの型です。</typeparam>
        /// <param name="obj">変換するオブジェクトです。</param>
        /// <param name="level">level のタブインデントを XmlElement 内部に設定します。</param>
        /// <returns>変換した XmlElement を返します。</returns>
        public static XmlElement ConvertObjectToXmlElement<T>(T obj, int level)
        {
            // インデントのレベルは1以上
            Ensure.Argument.True(level > 0);

            // タワーに変換
            var root = new TTower<T>();
            var leaf = root;

            for (int i = 1; i < level; i++)
            {
                leaf.Tower = new TTower<T>();
                leaf = leaf.Tower;
            }

            leaf.Data = obj;

            // シリアライズ
            var stream = new MemoryStream();
            XmlWriterSettings settings = new XmlWriterSettings();
            using (XmlWriter writer = XmlWriter.Create(stream, settings))
            {
                var serializer = GetXmlSerializer(typeof(TTower<T>));
                serializer.Serialize(writer, root);
            }

            stream.Seek(0, SeekOrigin.Begin);

            // 変換
            var doc = XDocument.Load(stream, LoadOptions.PreserveWhitespace);
            doc.Root.Name = typeof(XmlElementTower).Name;
            var element = doc.XPathSelectElement("//Data");
            element.Name = GetXmlElementName(typeof(T));

            // デシリアライズ
            XmlElementTower node;
            using (XmlReader reader = doc.CreateReader())
            {
                var deserializer = GetXmlSerializer(typeof(XmlElementTower));
                node = deserializer.Deserialize(reader) as XmlElementTower;
            }

            // タワーから要素を取り出します。
            while (node.Tower != null)
            {
                node = node.Tower;
            }

            return node.Elements[0];
        }

        /// <summary>
        /// オブジェクト を XmlElement に変換します。
        /// </summary>
        /// <param name="obj">変換するオブジェクトです。</param>
        /// <returns>変換した XmlElement を返します。</returns>
        public static XmlElement ConvertObjectToXmlElement(object obj)
        {
            Type type = obj.GetType();

            // シリアライズ
            var stream = new MemoryStream();
            XmlWriterSettings settings = new XmlWriterSettings();
            using (XmlWriter writer = XmlWriter.Create(stream, settings))
            {
                var xnameSpace = new XmlSerializerNamespaces();
                xnameSpace.Add(string.Empty, string.Empty);
                var serializer = GetXmlSerializer(type);
                serializer.Serialize(writer, obj, xnameSpace);
            }

            stream.Seek(0, SeekOrigin.Begin);

            // 変換
            var doc = XDocument.Load(stream, LoadOptions.PreserveWhitespace);
            doc.Root.Name = GetXmlElementName(type);

            // デシリアライズ
            XmlElement element;
            using (XmlReader reader = doc.CreateReader())
            {
                var deserializer = GetXmlSerializer(typeof(XmlElement));
                element = deserializer.Deserialize(reader) as XmlElement;
            }

            return element;
        }

        /// <summary>
        /// XmlElement を自動推測された型のオブジェクトに変換します。
        /// </summary>
        /// <param name="element">変換する XmlElement です。</param>
        /// <returns>変換したオブジェクトを返します。</returns>
        public static object ConvertXmlElementToObject(XmlElement element)
        {
            var type = GetTypeFromXmlElement(element);
            if (type == null)
            {
                return null;
            }

            return ConvertXmlElementToObject(element, type);
        }

        /// <summary>
        /// XmlElement を T に変換します。
        /// </summary>
        /// <typeparam name="T">変換するオブジェクトの型です。</typeparam>
        /// <param name="element">変換する XmlElement です。</param>
        /// <returns>変換したオブジェクトを返します。</returns>
        public static T ConvertXmlElementToObject<T>(XmlElement element) where T : class
        {
            return ConvertXmlElementToObject(element, typeof(T)) as T;
        }

        /// <summary>
        /// XmlElement を 特定の型のオブジェクト に変換します。
        /// </summary>
        /// <param name="element">変換する XmlElement です。</param>
        /// <param name="type">変換するオブジェクトの型です。</param>
        /// <returns>変換したオブジェクトを返します。</returns>
        public static object ConvertXmlElementToObject(XmlElement element, Type type)
        {
            var serializer = GetXmlSerializer(type);
            return serializer.Deserialize(new XmlNodeReader(element));
        }

        /// <summary>
        /// 特定の型のオブジェクトをシリアライズされた文字列に変換します。
        /// </summary>
        /// <typeparam name="T">シリアライズするオブジェクトの型を指定します。</typeparam>
        /// <param name="obj">シリアライズするオブジェクトです。</param>
        /// <returns>シリアライズ後の文字列を返します。</returns>
        public static string SerializeToString<T>(T obj)
            where T : new()
        {
            if (!IsSerializable(typeof(T)))
            {
                throw new InvalidOperationException("対象オブジェクトはシリアライズ可能な型である必要があります。");
            }

            XmlSerializer serializer = GetXmlSerializer(obj.GetType());

            using (StringWriter writer = new StringWriter())
            {
                serializer.Serialize(writer, obj);

                return writer.ToString();
            }
        }

        /// <summary>
        /// XML文字列を特定の型のオブジェクトにデシリアライズします。
        /// </summary>
        /// <typeparam name="T">デシリアライズ後の型を指定します。</typeparam>
        /// <param name="xml">デシリアライズするXML文字列を渡します。</param>
        /// <returns>デシリアライズされたオブジェクトを返します。失敗した場合はnullを返します。</returns>
        public static T DeserializeFromString<T>(string xml)
            where T : class, new()
        {
            return DeserializeFromString(xml, typeof(T)) as T;
        }

        /// <summary>
        /// XML文字列を特定の型のオブジェクトにデシリアライズします。
        /// </summary>
        /// <param name="xml">デシリアライズするXML文字列を渡します。</param>
        /// <param name="outputType">デシリアライズ後の型を指定します。</param>
        /// <returns>デシリアライズされたオブジェクトを返します。失敗した場合はnullを返します。</returns>
        public static object DeserializeFromString(string xml, Type outputType)
        {
            if (!IsSerializable(outputType))
            {
                return null;
            }

            XmlSerializer serializer = GetXmlSerializer(outputType);

            using (StringReader reader = new StringReader(xml))
            {
                try
                {
                    return serializer.Deserialize(reader);
                }
                catch (Exception)
                {
                    return null;
                }
            }
        }

        /// <summary>
        /// XML要素の名前を型から取得します。
        /// </summary>
        /// <param name="type">シリアライズ可能な型を指定します。</param>
        /// <returns>XML要素の名前を返します。</returns>
        public static string GetXmlElementName(Type type)
        {
            Ensure.Argument.True(type.IsSerializable);

            string xmlRootElementName = type.Name;
            var attributes = type.GetCustomAttributesData();
            foreach (var attribute in attributes)
            {
                if (attribute.AttributeType == typeof(XmlRootAttribute) && attribute.ConstructorArguments.Count > 0)
                {
                    var arg = attribute.ConstructorArguments[0];
                    Ensure.True<InvalidOperationException>(arg.ArgumentType == typeof(string));

                    xmlRootElementName = arg.Value as string;
                    break;
                }
            }

            return xmlRootElementName;
        }

        private static bool IsSerializable(Type type)
        {
            return type.IsSerializable || typeof(ISerializable).IsAssignableFrom(type);
        }

        private static Type GetTypeFromXmlElement(XmlElement element)
        {
            if (ElementNameToTypeDict.ContainsKey(element.Name))
            {
                return ElementNameToTypeDict[element.Name];
            }

            System.Reflection.Assembly[] assemblies = AppDomain.CurrentDomain.GetAssemblies();
            lock (LoadedAssemblies)
            {
                foreach (var asm in assemblies.Where(a => !LoadedAssemblies.Contains(a.FullName)))
                {
                    foreach (var type in asm.GetTypes().Where(t => t.IsSerializable))
                    {
                        var name = GetXmlElementName(type);
                        if (!ElementNameToTypeDict.ContainsKey(name))
                        {
                            ElementNameToTypeDict.TryAdd(name, type);
                        }

                        if (name == element.Name)
                        {
                            return type;
                        }
                    }
                    LoadedAssemblies.Add(asm.FullName);
                }
            }

            return null;
        }

        private static XmlSerializer GetXmlSerializer(Type type)
        {
            if (!XmlSrializerPool.ContainsKey(type.FullName))
            {
                var serializer = new XmlSerializer(type);
                XmlSrializerPool.TryAdd(type.FullName, serializer);
                return serializer;
            }

            return XmlSrializerPool[type.FullName];
        }

        /// <summary>
        /// 変換用ヘルパクラスです。
        /// </summary>
        /// <typeparam name="T">変換する型です。</typeparam>
        public class TTower<T>
        {
            /// <summary>
            /// TTower です。
            /// </summary>
            [XmlElement]
            public TTower<T> Tower { get; set; }

            /// <summary>
            /// データです。
            /// </summary>
            public T Data { get; set; }
        }

        /// <summary>
        /// 変換用ヘルパクラスです。
        /// </summary>
        public class XmlElementTower
        {
            /// <summary>
            /// XmlElementTower です。
            /// </summary>
            [XmlElement]
            public XmlElementTower Tower { get; set; }

            /// <summary>
            /// エレメントです。
            /// </summary>
            [XmlAnyElement]
            public XmlElement[] Elements { get; set; }
        }
    }
}
