﻿namespace Opal.Utilities
{
    using System;
    using System.Diagnostics;

    /// <summary>
    /// float 関連のユーティリティです。
    /// </summary>
    public static class FloatUtility
    {
        /// <summary>
        /// 許容誤差です。
        /// <remarks>
        /// 機械イプシロンの値です。
        /// (1.00.....X 1.0をfloatで表現した際の最下位桁の値。)
        /// IEEE754でfloatの仮数部ビットは23より計算して最下位桁は1.0e-7fとなる。
        /// --------------------------------------------------
        /// 以下[機械イプシロン]Googleヒットページより抜粋。
        /// --------------------------------------------------
        /// 与えられた浮動小数点数の中の最後の桁が持つ重みをulp(アルプ)といいます。
        /// (unit in the last place の略)
        /// ulpの大きさは指数によって決定されます。
        /// 浮動小数点数はこれと同じ程度の誤差を常に持ちます。
        /// 特に、1.0 の ulp を機械イプシロンと呼び、 これは相対誤差の大きさの基準として
        /// 用いられることがあります。
        /// </remarks>
        /// </summary>
        public const float Epsilon = 1.0e-7f;

        /// <summary>
        /// 2 つの float の値がほぼ等しいかどうかを示します。
        /// </summary>
        /// <remarks>
        /// Epsilon を差の許容値として利用します。
        /// </remarks>
        /// <param name="lhs">比較する左辺値です。</param>
        /// <param name="rhs">比較する右辺値です。</param>
        /// <returns>2 つの float の差が Epsilon 以下の場合は true を返します。
        /// それ以外の場合は false を返します。</returns>
        public static bool NearlyEqual(float lhs, float rhs)
        {
            return System.Math.Abs(lhs - rhs) <= Epsilon;
        }

        /// <summary>
        /// 2 つの float の値がほぼ等しいかどうかを相対許誤差で判定します。
        /// </summary>
        /// <remarks>
        /// 書籍：ゲームプログラミングのためのリアルタイム衝突判定(ISBN4-939007-91-X)
        /// を参考にしました。
        /// </remarks>
        /// <param name="lhs">比較する左辺値です。</param>
        /// <param name="rhs">比較する右辺値です。</param>
        /// <returns>2 つの float の比差が Epsilon 以下の場合は true を返します。
        /// それ以外の場合は false を返します。</returns>
        public static bool NearlyEqualInRelativeManner(float lhs, float rhs)
        {
            // Abs(x) < Abs(y) と仮定して、2値の比がどれだけ１に近いかという考え方から
            //  Math.Abs(x/y) - 1.0f) <= Epsilon
            // v
            //  Math.Abs(x/y) / y  <= Epsilon // 変形
            // v
            //  Math.Abs(x/y) / y  <= Epsilon // 分母にAbs(y)を乗算
            // v
            //  Math.Abs(x/y) <= Epsilon * Math.Abs(y) // 分母にAbs(y)を乗算
            // v
            //  Abs(x) < Abs(y) の仮定を外し、xやyが1.0より小さい場合も考慮すると下式
            // v
            return System.Math.Abs(lhs - rhs) <= Epsilon * Math.Max(Math.Max(lhs, rhs), 1.0f);
        }

        /// <summary>
        /// 2 つの float の値がほぼ等しいかどうかを示します。
        /// </summary>
        /// <param name="lhs">比較する左辺値です。</param>
        /// <param name="rhs">比較する右辺値です。</param>
        /// <param name="tolerance">差の許容値です。</param>
        /// <returns>2 つの float の差が tolerance 以下の場合は true を返します。
        /// それ以外の場合は false を返します。</returns>
        public static bool NearlyEqual(float lhs, float rhs, float tolerance)
        {
            return System.Math.Abs(lhs - rhs) <= tolerance;
        }

        /// <summary>
        /// 2 つの float の値が小数点以下の指定の桁数（切捨て）内で等しいかどうかを示します。
        /// </summary>
        /// <param name="lhs">比較する左辺値です。</param>
        /// <param name="rhs">比較する右辺値です。</param>
        /// <param name="precision">小数点以下の桁数です。</param>
        /// <returns>2 つの float の値が小数点以下の指定の桁数（切捨て）内で
        /// 等しい場合は true を返します。それ以外の場合は false を返します。</returns>
        public static bool NearlyEqualNumberOfSignificantFigures(
            float lhs, float rhs, int precision)
        {
            // TODO: 実装
            throw new NotImplementedException();
        }

        /// <summary>
        /// float の値を指定の小数点以下の桁数で切り上げを行います。
        /// </summary>
        /// <param name="v">float の値です。</param>
        /// <param name="presicion">小数点以下の桁数です。</param>
        /// <returns>切り上げの結果です。</returns>
        public static float RoundUp(float v, int presicion)
        {
            // TODO: 実装
            throw new NotImplementedException();
        }

        /// <summary>
        /// float の値を指定の小数点以下の桁数で切り捨てを行います。
        /// </summary>
        /// <param name="v">float の値です。</param>
        /// <param name="precision">小数点以下の桁数です。</param>
        /// <returns>切り上げの結果です。</returns>
        public static float RoundDown(float v, int precision)
        {
            // TODO: 実装
            throw new NotImplementedException();
        }

        /// <summary>
        /// 2 つの float の値を交換します。
        /// </summary>
        /// <param name="rhs">交換する左辺値です。</param>
        /// <param name="lhs">交換する右辺値です。</param>
        public static void Swap(ref float rhs, ref float lhs)
        {
            float temp = rhs;
            rhs = lhs;
            lhs = temp;
        }

        /// <summary>
        /// 指定した float の値以下の最大の整数を取得します。
        /// </summary>
        /// <param name="v">float の値です。</param>
        /// <returns>整数値です。</returns>
        public static int Floor(float v)
        {
            return (int)Math.Floor(v);
        }

        /// <summary>
        /// float 値を最大値と最小値の間にクランプします。
        /// </summary>
        /// <param name="value">クランプする値です。</param>
        /// <param name="minValue">最小値です。</param>
        /// <param name="maxValue">最大値です。</param>
        /// <returns>クランプした値です。</returns>
        public static float Clamp(float value, float minValue, float maxValue)
        {
            if (value < minValue)
            {
                return minValue;
            }

            if (value > maxValue)
            {
                return maxValue;
            }

            return value;
        }

        /// <summary>
        /// 値が指定された範囲内にあるか調べます。
        /// </summary>
        /// <param name="value">範囲チェックする値です。</param>
        /// <param name="min">最小値です。</param>
        /// <param name="max">最大値です。</param>
        /// <returns>範囲内であれば true を返します。</returns>
        public static bool Contains(float value, float min, float max)
        {
            Debug.Assert(min <= max);

            if (value < min)
            {
                return false;
            }

            if (max < value)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// 値を元の範囲から目的の範囲に範囲における位置の比率を保ちつつコンバートします。
        /// </summary>
        /// <param name="value">コンバート対象です。</param>
        /// <param name="srcMin">元の範囲の最小値です。</param>
        /// <param name="srcMmx">元の範囲の最大値です。</param>
        /// <param name="dstMin">目的の範囲の最小値です。</param>
        /// <param name="dstMax">目的の範囲の最大値です。</param>
        /// <returns>コンバートした結果です。</returns>
        public static float Convert(
            float value, float srcMin, float srcMmx, float dstMin, float dstMax)
        {
            // 動作例です。
            /*
            float delta = 0.0001f;
            Assert.AreEqual(20.0f, FloatUtility.Convert(0.2f, 0.0f, 1.0f, 0.0f, 100.0f), delta);
            Assert.AreEqual(120.0f, FloatUtility.Convert(0.2f, 0.0f, 1.0f, 100.0f, 200.0f), delta);
            Assert.AreEqual(120.0f, FloatUtility.Convert(1.2f, 1.0f, 2.0f, 100.0f, 200.0f), delta);
            Assert.AreEqual(0.0f, FloatUtility.Convert(0.0f, 0.0f, 10.0f, 0.0f, 20.0f), delta);
            Assert.AreEqual(10.0f, FloatUtility.Convert(5.0f, 0.0f, 10.0f, 0.0f, 20.0f), delta);
            Assert.AreEqual(-10.0f, FloatUtility.Convert(0.0f, 0.0f, 10.0f, -10.0f, 0.0f), delta);
            Assert.AreEqual(15.0f, FloatUtility.Convert(15.0f, 10.0f, 20.0f, 0.0f, 30.0f), delta);
            Assert.AreEqual(20.0f, FloatUtility.Convert(15.0f, 10.0f, 20.0f, 10.0f, 30.0f), delta);
            */
            return ((value - srcMin) * (dstMax - dstMin) / (srcMmx - srcMin)) + dstMin;
        }

        /// <summary>
        /// 値が0に近い場合0に丸めます。
        /// </summary>
        /// <param name="value">元の値です。</param>
        /// <returns>結果です。</returns>
        public static float RoundToZero(float value)
        {
            if (Contains(value, -Epsilon, Epsilon))
            {
                return 0.0F;
            }

            return value;
        }

        /// <summary>
        /// float 値のビット列を uint として取得します。
        /// </summary>
        /// <param name="value">float値です。</param>
        /// <returns>uintに読み替えた値です。</returns>
        public static uint FloatAsUInt(float value)
        {
            byte[] bytes = BitConverter.GetBytes(value);
            uint bits32 = BitConverter.ToUInt32(bytes, 0);

            return bits32;
        }

        /// <summary>
        /// uint 値のビット列を float として取得します。
        /// </summary>
        /// <param name="value">uint 値です。</param>
        /// <returns>float に読み替えた値です。</returns>
        public static float UintAsFloat(uint value)
        {
            byte[] bytes = BitConverter.GetBytes(value);
            float bits32 = BitConverter.ToSingle(bytes, 0);

            return bits32;
        }

        /// <summary>
        /// 32bit float から 4.12 の固定少数に変換します。
        /// </summary>
        /// <param name="value">float 値です。</param>
        /// <returns>16bit のビット表現です。</returns>
        public static uint Float32ToFixed4_12(float value)
        {
            const int NUMERIC_WIDTH = 4;
            const int FRACTION_WIDTH = 12;
            const int FIXED_WIDTH = 16;

            float floatValue = value;
            uint bits32 = FloatAsUInt(value);

            if ((floatValue == 0.0f) || ((bits32 & 0x7f800000) == 0x7f800000))
            {
                return 0;
            }
            else
            {
                floatValue += 0.5f * (1 << NUMERIC_WIDTH); // 負の数を正の値に底上げ。
                floatValue *= 1 << FRACTION_WIDTH;
                if (floatValue < 0)
                {
                    floatValue = 0;
                }
                else if (floatValue >= 1 << FIXED_WIDTH)
                {
                    floatValue = (1 << FIXED_WIDTH) - 1;
                }

                if (floatValue >= (1 << (FIXED_WIDTH - 1)))
                {
                    return (uint)floatValue - (1 << (FIXED_WIDTH - 1));
                }
                else
                {
                    return (uint)floatValue + (1 << (FIXED_WIDTH - 1));
                }
            }
        }
    }
}
