﻿namespace Opal.Storages
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using Opal.Utilities;

    /// <summary>
    /// ストレージ管理クラスです。
    /// </summary>
    public sealed class StorageManager : IStorageManager
    {
        private static readonly string ManagerKeyString;

        private readonly object syncRoot = new object();
        private readonly Dictionary<Type, Storage> cache =
            new Dictionary<Type, Storage>();

        static StorageManager()
        {
            ManagerKeyString = Enum.GetName(typeof(ManagerKey), ManagerKey.Storage);
        }

        /// <summary>
        /// マネージャーのキーを取得します。
        /// </summary>
        public string Key
        {
            get
            {
                return ManagerKeyString;
            }
        }

        /// <summary>
        /// ストレージを取得します。
        /// </summary>
        /// <typeparam name="TStorage">ストレージのテンプレートの型です。</typeparam>
        /// <returns>指定したグループに存在するストレージのインスタンスを返します。</returns>
        public WeakReference<TStorage> GetStorage<TStorage>() where TStorage : Storage
        {
            lock (this.syncRoot)
            {
                Type type = typeof(TStorage);
                WeakReference<TStorage> weakStorage = new WeakReference<TStorage>((TStorage)this.cache[type]);
                return weakStorage;
            }
        }

        /// <summary>
        /// ストレージを取得を試行します。
        /// </summary>
        /// <typeparam name="TStorage">ストレージのテンプレートの型です。</typeparam>
        /// <param name="storage">取得成功ならば、ストレージのインスタンスを格納します。存在しない場合は、null を格納します。</param>
        /// <returns>取得成功の場合は、true を失敗の場合は、false を返します。</returns>
        public bool TryGetStorage<TStorage>(out WeakReference<TStorage> storage) where TStorage : Storage
        {
            lock (this.syncRoot)
            {
                Storage result = null;
                Type type = typeof(TStorage);
                if (this.cache.TryGetValue(type, out result))
                {
                    storage = new WeakReference<TStorage>((TStorage)result);
                    return true;
                }

                storage = null;
                return false;
            }
        }

        /// <summary>
        /// ストレージを所有しているか判定します。
        /// </summary>
        /// <typeparam name="TStorage">ストレージのテンプレートの型です。</typeparam>
        /// <returns>所有している場合は、true 所有していない場合は、false を返します。</returns>
        public bool HasStorage<TStorage>() where TStorage : Storage
        {
            lock (this.syncRoot)
            {
                Type type = typeof(TStorage);
                return this.cache.ContainsKey(type);
            }
        }

        /// <summary>
        /// ストレージを追加します。
        /// </summary>
        /// <typeparam name="TStorage">ストレージのテンプレートの型です。</typeparam>
        /// <param name="storage">追加するストレージです。</param>
        public void AddStorage<TStorage>(TStorage storage) where TStorage : Storage
        {
            lock (this.syncRoot)
            {
                Type type = typeof(TStorage);
                Storage resultStorage = null;
                bool result = this.cache.TryGetValue(type, out resultStorage);
                Debug.Assert(!result);
                this.cache.Add(type, storage);
            }
        }

        /// <summary>
        /// ストレージを削除します。
        /// </summary>
        /// <typeparam name="TStorage">ストレージのテンプレートの型です。</typeparam>
        public void RemoveStorage<TStorage>() where TStorage : Storage
        {
            lock (this.syncRoot)
            {
                Type type = typeof(TStorage);
                Storage resultStorage = null;
                bool result = this.cache.TryGetValue(type, out resultStorage);
                Debug.Assert(result);
                this.cache.Remove(type);
                resultStorage.Dispose();
            }
        }

        /// <summary>
        /// 状態をクリアします。
        /// </summary>
        public void Clear()
        {
            lock (this.syncRoot)
            {
                this.cache.Clear();
            }
        }

        /// <summary>
        /// ストレージを取得します。
        /// </summary>
        /// <returns>ストレージのインスタンスを返します。</returns>
        internal IEnumerable<Storage> GetStorages()
        {
            foreach (var storage in this.cache.Values)
            {
                yield return storage;
            }
        }
    }
}
