﻿namespace Opal.Panes
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using System.Windows.Data;
    using Opal.ComponentModel;
    using Opal.Utilities;
    using Opal.ViewModels;

    /// <summary>
    /// ペイン管理クラスです。
    /// </summary>
    public sealed class PaneManager : DisposableObservableObject, IPaneManager
    {
        private static readonly string ManagerKeyString;

        private ObservableCollection<DocumentViewModel> documents =
            new ObservableCollection<DocumentViewModel>();

        private ObservableCollection<ToolViewModel> tools =
            new ObservableCollection<ToolViewModel>();

        static PaneManager()
        {
            ManagerKeyString = Enum.GetName(typeof(ManagerKey), ManagerKey.Pane);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PaneManager()
        {
            BindingOperations.EnableCollectionSynchronization(this.documents, this.SyncRoot);
            BindingOperations.EnableCollectionSynchronization(this.tools, this.SyncRoot);
        }

        /// <summary>
        /// マネージャーのキーを取得します。
        /// </summary>
        public string Key
        {
            get
            {
                return ManagerKeyString;
            }
        }

        /// <summary>
        /// ドキュメントを取得します。
        /// </summary>
        /// <returns>ドキュメントのインスタンスを返します。</returns>
        public ICollectionView GetDocuments()
        {
            return CollectionViewSource.GetDefaultView(this.documents);
        }

        /// <summary>
        /// ツールを取得します。
        /// </summary>
        /// <returns>ツールのインスタンスを返します。</returns>
        public ICollectionView GetTools()
        {
            return CollectionViewSource.GetDefaultView(this.tools);
        }

        /// <summary>
        /// ドキュメントを追加します。
        /// </summary>
        /// <param name="viewModel">追加するドキュメントです。</param>
        public void AddDocument(DocumentViewModel viewModel)
        {
            Debug.Assert(viewModel != null);
            lock (this.SyncRoot)
            {
                if (!this.documents.Contains(viewModel))
                {
                    this.documents.Add(viewModel);
                }
            }
        }

        /// <summary>
        /// ツールを追加します。
        /// </summary>
        /// <param name="viewModel">追加するツールです。</param>
        public void AddTool(ToolViewModel viewModel)
        {
            Debug.Assert(viewModel != null);
            lock (this.SyncRoot)
            {
                if (!this.tools.Contains(viewModel))
                {
                    this.tools.Add(viewModel);
                }
            }
        }

        /// <summary>
        /// ドキュメントを削除します。
        /// </summary>
        /// <param name="viewModel">削除するドキュメントです。</param>
        public void RemoveDocument(DocumentViewModel viewModel)
        {
            Debug.Assert(viewModel != null);
            lock (this.SyncRoot)
            {
                if (this.documents.Contains(viewModel))
                {
                    this.documents.Remove(viewModel);
                }
            }
        }

        /// <summary>
        /// ツールを削除します。
        /// </summary>
        /// <param name="viewModel">削除するツールです。</param>
        public void RemoveTool(ToolViewModel viewModel)
        {
            Debug.Assert(viewModel != null);
            lock (this.SyncRoot)
            {
                if (this.tools.Contains(viewModel))
                {
                    this.tools.Remove(viewModel);
                }
            }
        }

        /// <summary>
        /// 指定ドキュメントを含んでいるか判定します。
        /// </summary>
        /// <param name="viewModel">判定対象のドキュメントです。</param>
        /// <returns>含んでいる場合は、true を返します。</returns>
        public bool ContainDocument(DocumentViewModel viewModel)
        {
            Debug.Assert(viewModel != null);
            lock (this.SyncRoot)
            {
                return this.documents.Contains(viewModel);
            }
        }

        /// <summary>
        /// 指定ツールを含んでいるか判定します。
        /// </summary>
        /// <param name="viewModel">判定対象のツールです。</param>
        /// <returns>含んでいる場合は、true を返します。</returns>
        public bool ContainTool(ToolViewModel viewModel)
        {
            Debug.Assert(viewModel != null);
            lock (this.SyncRoot)
            {
                return this.tools.Contains(viewModel);
            }
        }

        /// <summary>
        /// クリアします。
        /// </summary>
        public void Clear()
        {
            this.tools.Clear();
            this.documents.Clear();
        }

        /// <summary>
        /// オブジェクト破棄の内部処理です。継承した先で固有の処理を実装します。
        /// </summary>
        protected override void DisposeInternal()
        {
            this.tools.Clear();
            this.documents.Clear();

            BindingOperations.DisableCollectionSynchronization(this.tools);
            BindingOperations.DisableCollectionSynchronization(this.documents);
        }
    }
}
