﻿namespace Opal.Operations
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using System.Windows.Data;
    using Opal.App;
    using Opal.ComponentModel;
    using Opal.Utilities;

    /// <summary>
    /// オペレーション管理クラスです。
    /// </summary>
    public sealed class OperationManager : DisposableObservableObject, IOperationManager
    {
        private static readonly string ManagerKeyString;
        private readonly ObservableCollection<Operation> undoStack = new ObservableCollection<Operation>();
        private readonly ObservableCollection<Operation> redoStack = new ObservableCollection<Operation>();
        private int undoMaxCount = 30;
        private OperationTransaction transaction = null;

        static OperationManager()
        {
            ManagerKeyString = Enum.GetName(typeof(ManagerKey), ManagerKey.Operation);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public OperationManager()
        {
            BindingOperations.EnableCollectionSynchronization(this.undoStack, this.SyncRoot);
            BindingOperations.EnableCollectionSynchronization(this.redoStack, this.SyncRoot);
        }

        /// <summary>
        /// マネージャーのキーを取得します。
        /// </summary>
        public string Key
        {
            get
            {
                return ManagerKeyString;
            }
        }

        /// <summary>
        /// アンドゥスタックを読み取り専用で取得します。
        /// </summary>
        public IReadOnlyList<Operation> UndoStack
        {
            get
            {
                return this.undoStack;
            }
        }

        /// <summary>
        /// リドゥスタックを読み取り専用で取得します。
        /// </summary>
        public IReadOnlyList<Operation> RedoStack
        {
            get
            {
                return this.redoStack;
            }
        }

        /// <summary>
        /// トランザクションを開始します。
        /// </summary>
        /// <param name="merge">trueの場合は、マージ可能なオペレーションをマージします。</param>
        /// <returns>トランザクションのインスタンスを返します。</returns>
        public OperationTransaction BeginTransaction(bool merge)
        {
            lock (this.SyncRoot)
            {
                if (this.transaction == null)
                {
                    this.transaction = new OperationTransaction(this, merge);
                }

                return this.transaction;
            }
        }

        /// <summary>
        /// トランザクションを終了します。
        /// </summary>
        public void EndTransaction()
        {
            lock (this.SyncRoot)
            {
                if (this.transaction != null)
                {
                    OperationSet operationSet = new OperationSet();
                    operationSet.AddRange(this.transaction.Operations);
                    this.transaction = null;
                    this.Execute(operationSet);
                }
            }
        }

        /// <summary>
        /// Undo が可能か判定します。
        /// </summary>
        /// <returns>Undo が可能ならば true です。</returns>
        public bool CanUndo()
        {
            lock (this.SyncRoot)
            {
                return this.undoStack.Count > 0;
            }
        }

        /// <summary>
        /// Redo が可能か判定します。
        /// </summary>
        /// <returns>Redo が可能ならば true です。</returns>
        public bool CanRedo()
        {
            lock (this.SyncRoot)
            {
                return this.redoStack.Count > 0;
            }
        }

        /// <summary>
        /// オペレーションを追加します。
        /// </summary>
        /// <param name="operation">追加するオペレーションです。</param>
        public void Push(Operation operation)
        {
            if (!operation.CanUndoable())
            {
                return;
            }

            lock (this.SyncRoot)
            {
                this.redoStack.Clear();
                this.undoStack.Add(operation);
                while (this.undoStack.Count > this.undoMaxCount)
                {
                    this.undoStack.RemoveAt(0);
                }
            }
        }

        /// <summary>
        /// クリアします。
        /// </summary>
        public void Clear()
        {
            lock (this.SyncRoot)
            {
                this.undoStack.Clear();
                this.redoStack.Clear();
                this.transaction = null;
                Debug.WriteLine("undoStack cleared");
            }
        }

        /// <summary>
        /// 実行します。
        /// </summary>
        /// <param name="operation">実行するオペレーションです。</param>
        public void Execute(Operation operation)
        {
            operation = operation.Execute();
            if (!operation.CanUndoable())
            {
                return;
            }

            lock (this.SyncRoot)
            {
                if (this.transaction != null)
                {
                    this.transaction.Add(operation);
                }
                else
                {
                    this.Push(operation);
                }
            }
        }

        /// <summary>
        /// Redo します。
        /// </summary>
        public void Redo()
        {
            Debug.Assert(this.CanRedo());
            lock (this.SyncRoot)
            {
                int lastIndex = this.redoStack.Count - 1;
                Operation operation = this.redoStack[lastIndex];
                this.redoStack.RemoveAt(lastIndex);
                this.undoStack.Add(operation.Execute());
            }
        }

        /// <summary>
        /// Undo します。
        /// </summary>
        public void Undo()
        {
            Debug.Assert(this.CanUndo());
            lock (this.SyncRoot)
            {
                int lastIndex = this.undoStack.Count - 1;
                Operation operation = this.undoStack[lastIndex];
                this.undoStack.RemoveAt(lastIndex);
                this.redoStack.Add(operation.Execute());
            }
        }

        /// <summary>
        /// オブジェクト破棄の内部処理です。継承した先で固有の処理を実装します。
        /// </summary>
        protected override void DisposeInternal()
        {
            this.undoStack.Clear();
            this.redoStack.Clear();

            BindingOperations.DisableCollectionSynchronization(this.undoStack);
            BindingOperations.DisableCollectionSynchronization(this.redoStack);
        }
    }
}
