﻿namespace Opal.Operations
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using Opal.Utilities;

    /// <summary>
    /// リストソートオペレーションです。
    /// </summary>
    /// <typeparam name="TObject">リストアイテムのテンプレート型です。</typeparam>
    /// <typeparam name="TKey">ソートキーのテンプレート型です。</typeparam>
    public sealed class ListSortOperation<TObject, TKey> : Operation
    {
        private readonly IList<TObject> target;
        private readonly List<TObject> sourceList = new List<TObject>();
        private Func<TObject, TKey> keySelector = null;
        private bool undoFlag = false;
        private bool isDescendingOrder = false;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="keySelector">キーセレクタです。</param>
        /// <param name="isDescendingOrder">降順の場合はtrueを指定します。</param>
        public ListSortOperation(IList<TObject> target, Func<TObject, TKey> keySelector, bool isDescendingOrder = false)
        {
            this.keySelector = keySelector;
            this.target = target;
            this.isDescendingOrder = isDescendingOrder;
        }

        /// <summary>
        /// マージ可能か判定します。
        /// </summary>
        public override bool IsMergeable
        {
            get
            {
                return false;
            }
        }

        /// <summary>
        /// オペレーションを実行します。
        /// </summary>
        /// <returns>実行時に返される Operation です。</returns>
        public override Operation Execute()
        {
            if (this.undoFlag)
            {
                this.Undo();
            }
            else
            {
                this.sourceList.Clear();
                this.sourceList.AddRange(this.target);
                this.Redo();
            }

            this.undoFlag = !this.undoFlag;

            return this;
        }

        /// <summary>
        /// Undo を実行します。
        /// </summary>
        private void Undo()
        {
            this.target.Clear();
            foreach (var item in this.sourceList)
            {
                this.target.Add(item);
            }
        }

        /// <summary>
        /// Redo を実行します。
        /// </summary>
        private void Redo()
        {
            List<TObject> tempList = new List<TObject>();
            if (this.isDescendingOrder)
            {
                IOrderedEnumerable<TObject> ordered = this.target.OrderByDescending(this.keySelector);
                foreach (var item in ordered)
                {
                    tempList.Add(item);
                }
            }
            else
            {
                IOrderedEnumerable<TObject> ordered = this.target.OrderBy(this.keySelector);
                foreach (var item in ordered)
                {
                    tempList.Add(item);
                }
            }

            this.target.Clear();
            foreach (var item in tempList)
            {
                this.target.Add(item);
            }
        }
    }
}
