﻿namespace Opal.Operations
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;

    /// <summary>
    /// リスト系オペレーションの基底クラスです。
    /// </summary>
    /// <typeparam name="TObject">リストアイテムのテンプレート型です。</typeparam>
    public abstract class ListItemOperation<TObject> : Operation
    {
        private readonly IList<TObject> target;
        private readonly TObject item;
        private bool undoFlag = false;
        private bool isValueType = false;
        private int index = -1;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="item">対象のアイテムです。</param>
        protected ListItemOperation(IList<TObject> target, TObject item)
        {
            Debug.Assert(target != null);
            Debug.Assert(item != null);

            this.isValueType = item.GetType().IsValueType;
            this.target = target;
            this.item = item;

            if (this.isValueType)
            {
                int index = 0;
                foreach (var listItem in target)
                {
                    if (object.ReferenceEquals(listItem, item))
                    {
                        this.index = index;
                        break;
                    }

                    ++index;
                }
            }
        }

        /// <summary>
        /// マージ可能か判定します。
        /// </summary>
        public override bool IsMergeable
        {
            get
            {
                return false;
            }
        }

        /// <summary>
        /// 値型かどうかの状態を取得します。
        /// </summary>
        protected bool IsValueType
        {
            get
            {
                return this.isValueType;
            }
        }

        /// <summary>
        /// オペレーションを実行します。
        /// </summary>
        /// <returns>実行時に返される Operation です。</returns>
        public override Operation Execute()
        {
            if (this.undoFlag)
            {
                this.Undo(this.target, this.item);
            }
            else
            {
                this.Redo(this.target, this.item);
            }

            this.undoFlag = !this.undoFlag;

            return this;
        }

        /// <summary>
        /// Undo を実行します。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="item">対象のアイテムです。</param>
        protected abstract void Undo(IList<TObject> target, TObject item);

        /// <summary>
        /// Redo を実行します。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="item">対象のアイテムです。</param>
        protected abstract void Redo(IList<TObject> target, TObject item);
    }
}
