﻿namespace Opal.Menus
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using System.Windows.Data;
    using Opal.ComponentModel;
    using Opal.Utilities;

    /// <summary>
    /// メニュー管理クラスです。
    /// </summary>
    public sealed class MenuManager : DisposableObservableObject, IMenuManager
    {
        private static readonly string ManagerKeyString;

        private readonly Dictionary<MenuCategory, ObservableCollection<Menu>> menuCategories =
            new Dictionary<MenuCategory, ObservableCollection<Menu>>();

        private readonly ObservableCollection<Menu> displayMenus =
            new ObservableCollection<Menu>();

        static MenuManager()
        {
            ManagerKeyString = Enum.GetName(typeof(ManagerKey), ManagerKey.Menu);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public MenuManager()
        {
            foreach (var type in Enum.GetValues(typeof(MenuCategory)))
            {
                var menus = new ObservableCollection<Menu>();
                this.menuCategories.Add((MenuCategory)type, menus);
                BindingOperations.EnableCollectionSynchronization(menus, this.SyncRoot);
            }

            BindingOperations.EnableCollectionSynchronization(this.displayMenus, this.SyncRoot);
        }

        /// <summary>
        /// マネージャーのキーを取得します。
        /// </summary>
        public string Key
        {
            get
            {
                return ManagerKeyString;
            }
        }

        /// <summary>
        /// 表示メニューを取得します。
        /// </summary>
        /// <returns>表示メニューのインスタンスを返します。</returns>
        public ICollectionView GetDisplayMenus()
        {
            return CollectionViewSource.GetDefaultView(this.displayMenus);
        }

        /// <summary>
        /// メニューを取得します。
        /// </summary>
        /// <param name="category">取得対象のメニューの種類です。</param>
        /// <returns>メニューのインスタンスを返します。</returns>
        public ICollectionView GetMenus(MenuCategory category)
        {
            return CollectionViewSource.GetDefaultView(this.menuCategories[category]);
        }

        /// <summary>
        /// 表示メニューを追加します。
        /// </summary>
        /// <param name="menu">追加するメニューです。</param>
        public void AddDisplayMenu(Menu menu)
        {
            Debug.Assert(menu != null);
            lock (this.SyncRoot)
            {
                if (!this.displayMenus.Contains(menu))
                {
                    this.displayMenus.Add(menu);
                }
            }
        }

        /// <summary>
        /// メニューを追加します。
        /// </summary>
        /// <param name="category">追加対象のメニューの種類です。</param>
        /// <param name="menu">追加するメニューです。</param>
        public void AddMenu(MenuCategory category, Menu menu)
        {
            Debug.Assert(menu != null);
            lock (this.SyncRoot)
            {
                var menus = this.menuCategories[category];
                if (!menus.Contains(menu))
                {
                    menus.Add(menu);
                }
            }
        }

        /// <summary>
        /// メニューを削除します。
        /// </summary>
        /// <param name="category">削除対象のメニューの種類です。</param>
        /// <param name="menu">削除するメニューです。</param>
        public void RemoveMenu(MenuCategory category, Menu menu)
        {
            Debug.Assert(menu != null);
            lock (this.SyncRoot)
            {
                var menus = this.menuCategories[category];
                if (menus.Contains(menu))
                {
                    menus.Remove(menu);
                }
            }
        }

        /// <summary>
        /// クリアします。
        /// </summary>
        public void Clear()
        {
            foreach (var menus in this.menuCategories.Values)
            {
                menus.Clear();
            }

            this.displayMenus.Clear();
        }

        /// <summary>
        /// オブジェクト破棄の内部処理です。継承した先で固有の処理を実装します。
        /// </summary>
        protected override void DisposeInternal()
        {
            foreach (var menus in this.menuCategories.Values)
            {
                menus.Clear();
                BindingOperations.DisableCollectionSynchronization(menus);
            }

            this.menuCategories.Clear();

            this.displayMenus.Clear();
            BindingOperations.DisableCollectionSynchronization(this.displayMenus);
        }
    }
}
