﻿namespace Opal.Menus
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using System.Windows.Input;
    using Opal.Commands;
    using Opal.ComponentModel;
    using Opal.Utilities;

    /// <summary>
    /// コマンド実行付きメニュークラスです。
    /// </summary>
    public class CommandableMenu : Menu
    {
        private readonly string key;
        private Action<object> execute;
        private Func<object, bool> canExecute;
        private RelayCommand command = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="key">メニューのキーです。</param>
        /// <param name="label">メニューラベル名です。</param>
        public CommandableMenu(string key, string label)
            : base(label)
        {
            Debug.Assert(!string.IsNullOrWhiteSpace(key));
            Debug.Assert(key.Length >= 3);

            this.key = key;
            this.execute = param => { };
            this.canExecute = param => { return true; };
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="key">メニューのキーです。</param>
        /// <param name="label">メニューラベル名です。</param>
        /// <param name="execute">実行時のデリゲートです。</param>
        /// <param name="canExecute">実行可能か判定するデリゲートです。</param>
        public CommandableMenu(string key, string label, Action<object> execute, Func<object, bool> canExecute)
            : base(label)
        {
            Debug.Assert(!string.IsNullOrWhiteSpace(key));
            Debug.Assert(key.Length >= 3);

            this.key = key;
            this.execute = execute;
            this.canExecute = canExecute;
        }

        /// <summary>
        /// メニューコマンドを取得します。
        /// </summary>
        public ICommand Command
        {
            get
            {
                if (this.command == null)
                {
                    this.command = new RelayCommand(this.execute, this.canExecute);
                }

                return this.command;
            }
        }

        /// <summary>
        /// コマンドを設定します。
        /// </summary>
        /// <param name="execute">処理のデリゲートです。</param>
        /// <param name="canExecute">処理が可能か判定するデリゲートです。</param>
        protected void SetCommand(Action<object> execute, Func<object, bool> canExecute)
        {
            Debug.Assert(this.command == null);

            this.execute = execute;
            this.canExecute = canExecute;
        }
    }
}
