﻿namespace Opal.Configurations
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using System.Xml;
    using System.Xml.Serialization;
    using Opal.ComponentModel;
    using Opal.Services;
    using Opal.Utilities;
    using Opal.ViewModels;

    /// <summary>
    /// コンフィグクラスです。
    /// </summary>
    public abstract class Config : DisposableObservableObject
    {
        /// <summary>
        /// コンフィグファイルのバージョンを取得または設定します。
        /// </summary>
        [XmlAttribute]
        public string Version { get; set; }

        /// <summary>
        /// コンフィグファイルをロードします。
        /// </summary>
        /// <typeparam name="TConfig">コンフィグのテンプレートの型です。</typeparam>
        /// <param name="path">コンフィグファイルのパスです。</param>
        /// <returns>ロードしたコンフィグを返します。</returns>
        public static TConfig Load<TConfig>(string path) where TConfig : Config
        {
            return Deserialize<TConfig>(path);
        }

        /// <summary>
        /// コンフィグファイルを保存します。
        /// </summary>
        /// <typeparam name="TConfig">コンフィグのテンプレートの型です。</typeparam>
        /// <param name="config">保存するコンフィグです。</param>
        /// <param name="path">コンフィグファイルのパスです。</param>
        public static void Save<TConfig>(TConfig config, string path) where TConfig : Config
        {
            using (FileStream fileStream = new FileStream(path, FileMode.Create))
            {
                var xmlWriterSettings = new XmlWriterSettings()
                {
                    Encoding = new UTF8Encoding(true),  // BOM付き
                    Indent = true,
                    IndentChars = "\t",
                    CloseOutput = false,
                };

                using (var xmlWriter = XmlTextWriter.Create(fileStream, xmlWriterSettings))
                {
                    var nameSpace = new XmlSerializerNamespaces();
                    nameSpace.Add(string.Empty, string.Empty);

                    (new XmlSerializer(typeof(TConfig), string.Empty)).Serialize(xmlWriter, config, nameSpace);
                }
            }
        }

        /// <summary>
        /// コンフィグファイルを保存します。
        /// </summary>
        public virtual void Save()
        {
        }

        private static T Deserialize<T>(string filePath) where T : class
        {
            if (File.Exists(filePath))
            {
                using (FileStream fileStream = new FileStream(filePath, FileMode.Open, FileAccess.Read))
                {
                    return (T)(new XmlSerializer(typeof(T))).Deserialize(fileStream);
                }
            }

            return null;
        }
    }
}
