﻿// ========================================================================
// <copyright file="LayoutInitializer.cs" company="Nintendo">
//      Copyright 2013 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace Opal.Layout
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using System.Windows;
    using System.Windows.Controls;
    using Opal.Services;
    using Xceed.Wpf.AvalonDock.Layout;

    /// <summary>
    /// ドッキングウィンドウのレイアウト初期化クラスです。
    /// </summary>
    public sealed class LayoutInitializer : ILayoutUpdateStrategy
    {
        private enum InsertPosition
        {
            Start,
            End
        }

        /// <summary>
        /// アンカー追加前処理です。
        /// </summary>
        /// <param name="layout">レイアウトです。</param>
        /// <param name="anchorableToShow">アンカーです。</param>
        /// <param name="destinationContainer">レイアウトコンテナです。</param>
        /// <returns>処理成功時は、true を返します。</returns>
        public bool BeforeInsertAnchorable(LayoutRoot layout, LayoutAnchorable anchorableToShow, ILayoutContainer destinationContainer)
        {
            var tool = anchorableToShow.Content as ITool;
            if (tool != null)
            {
                var preferredLocation = tool.PreferredLocation;
                string paneName = GetPaneName(preferredLocation);
                var toolsPane = layout.Descendents().OfType<LayoutAnchorablePane>().FirstOrDefault(d => d.Name == paneName);
                if (toolsPane == null)
                {
                    switch (preferredLocation)
                    {
                        case PaneLocation.Left:
                            toolsPane = CreateAnchorablePane(layout, Orientation.Horizontal, paneName, InsertPosition.Start);
                            break;
                        case PaneLocation.Right:
                            toolsPane = CreateAnchorablePane(layout, Orientation.Horizontal, paneName, InsertPosition.End);
                            break;
                        case PaneLocation.Bottom:
                            toolsPane = CreateAnchorablePane(layout, Orientation.Vertical, paneName, InsertPosition.End);
                            break;
                        default:
                            toolsPane = CreateAnchorablePane(layout, Orientation.Horizontal, paneName, InsertPosition.End);
                            break;
                    }
                }

                toolsPane.Children.Add(anchorableToShow);
                return true;
            }

            return false;
        }

        /// <summary>
        /// アンカー追加後処理です。
        /// </summary>
        /// <param name="layout">レイアウトです。</param>
        /// <param name="anchorableShown">アンカーです。</param>
        public void AfterInsertAnchorable(LayoutRoot layout, LayoutAnchorable anchorableShown)
        {
            var tool = anchorableShown.Content as ITool;
            if (tool != null)
            {
                var anchorablePane = anchorableShown.Parent as LayoutAnchorablePane;
                if (anchorablePane != null && anchorablePane.ChildrenCount == 1)
                {
                    switch (tool.PreferredLocation)
                    {
                        case PaneLocation.Left:
                        case PaneLocation.Right:
                            anchorablePane.DockWidth = new GridLength(tool.PreferredWidth, GridUnitType.Pixel);
                            break;
                        case PaneLocation.Bottom:
                            anchorablePane.DockHeight = new GridLength(tool.PreferredHeight, GridUnitType.Pixel);
                            break;
                        default:
                            anchorablePane.DockWidth = new GridLength(tool.PreferredWidth, GridUnitType.Pixel);
                            break;
                    }
                }
            }
        }

        /// <summary>
        /// ドキュメント追加前処理です。
        /// </summary>
        /// <param name="layout">レイアウトです。</param>
        /// <param name="anchorableToShow">レイアウトドキュメントです。</param>
        /// <param name="destinationContainer">レイアウトコンテナです。</param>
        /// <returns>処理成功時にtrue を返します。</returns>
        public bool BeforeInsertDocument(LayoutRoot layout, LayoutDocument anchorableToShow, ILayoutContainer destinationContainer)
        {
            return false;
        }

        /// <summary>
        /// ドキュメント追加後処理です。
        /// </summary>
        /// <param name="layout">レイアウトです。</param>
        /// <param name="anchorableShown">レイアウトドキュメントです。</param>
        public void AfterInsertDocument(LayoutRoot layout, LayoutDocument anchorableShown)
        {
        }

        private static LayoutAnchorablePane CreateAnchorablePane(
            LayoutRoot layout,
            Orientation orientation,
            string paneName,
            InsertPosition position)
        {
            var parent = layout.Descendents().OfType<LayoutPanel>().First();
            Debug.Assert(parent != null);

            bool isFound = false;
            foreach (var layoutPanel in layout.Descendents().OfType<LayoutPanel>())
            {
                if (layoutPanel.Orientation == orientation)
                {
                    parent = layoutPanel;
                    isFound = true;
                    break;
                }
            }

            var toolsPane = new LayoutAnchorablePane { Name = paneName };
            if (position == InsertPosition.Start || !isFound)
            {
                parent.InsertChildAt(0, toolsPane);
            }
            else
            {
                parent.Children.Add(toolsPane);
            }

            return toolsPane;
        }

        private static string GetPaneName(PaneLocation location)
        {
            switch (location)
            {
                case PaneLocation.Left:
                    return "LeftPane";
                case PaneLocation.Right:
                    return "RightPane";
                case PaneLocation.Bottom:
                    return "BottomPane";
                default:
                    return "RightPane";
            }
        }
    }
}
