﻿using Nintendo.G3dTool.Entities;
using Nintendo.G3dTool.Entities.Internal;
using Nintendo.ToolFoundation.Contracts;
using nw.g3d.nw4f_3dif;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.G3dTool.Extensions
{
    public static class ShapeExtensions
    {
        private class VertexAttrInfo
        {
            public VertexAttrInfo(string name, string hint, vtx_attrib_typeType type)
            {
                this.NamePrefix = name;
                this.HintPrefix = hint;
                this.Type = type;
            }

            public string NamePrefix { get; set; }
            public string HintPrefix { get; set; }
            public vtx_attrib_typeType Type { get; set; }
        }

        private readonly static VertexAttrInfo PositionAttrInfo = new VertexAttrInfo("_p", "position", vtx_attrib_typeType.float3);
        private readonly static VertexAttrInfo ColorAttrInfo = new VertexAttrInfo("_c", "color", vtx_attrib_typeType.float4);
        private readonly static VertexAttrInfo NormalAttrInfo = new VertexAttrInfo("_n", "normal", vtx_attrib_typeType.float3);
        private readonly static VertexAttrInfo TangentAttrInfo = new VertexAttrInfo("_t", "tangent", vtx_attrib_typeType.float3);
        private readonly static VertexAttrInfo BinormalAttrInfo = new VertexAttrInfo("_b", "binormal", vtx_attrib_typeType.float3);
        private readonly static VertexAttrInfo UvAttrInfo = new VertexAttrInfo("_u", "uv", vtx_attrib_typeType.float2);
        private readonly static VertexAttrInfo BlendWeightAttrInfo = new VertexAttrInfo("_b", "blendweight", vtx_attrib_typeType.float4);
        private readonly static VertexAttrInfo BlendIndexAttrInfo = new VertexAttrInfo("_i", "blendindex", vtx_attrib_typeType.int4);

        /// <summary>
        /// 名前やインデックスで間接的に参照しているデータを含めてディープコピーします。
        /// </summary>
        /// <param name="shape"></param>
        /// <returns></returns>
        public static Shape DeepCopyWithReferences(this Shape shape, Model parentModel)
        {
            var copiedShape = new Shape(shape);
            parentModel.Shapes.Add(copiedShape);
            for (int i = 0; i < shape.Meshes.Count; ++i)
            {
                var mesh = shape.Meshes[i];
                var copiedMesh = copiedShape.Meshes[i];
                copiedMesh.IndexStream = new StreamInt(mesh.IndexStream);
                parentModel.Streams.Add(copiedMesh.IndexStream);
            }

            copiedShape.ShapeInfo.Vertex = shape.ShapeInfo.Vertex.DeepCopyWithReferences(parentModel);

            // ボーンも参照しているが、1対1で紐づくものではないので、コピーは行わない
            //copiedShape.ShapeInfo.Bone = shape.ShapeInfo.Bone.DeepCopyWithReferences(parentModel.Skeleton);

            return copiedShape;
        }

        public static VtxAttrib AddVertexAttribute(
            this Shape shape,
            string name,
            string hint,
            vtx_attrib_typeType type,
            Stream stream)
        {
            Ensure.Argument.True(shape.Parent.Streams.Contains(stream));

            var attr = new VtxAttrib();
            attr.Name = name;
            attr.Hint = hint;
            attr.Type = type;
            attr.Stream = stream;
            shape.ShapeInfo.Vertex.VtxAttribs.Add(attr);

            stream.Column = Utility.GetComponentCount(type);
            return attr;
        }

        public static VtxAttrib AddVertexAttributePosition(this Shape shape, StreamFloat stream)
        {
            return AddVertexAttribute(shape, PositionAttrInfo, stream);
        }

        public static VtxAttrib AddVertexAttributeColor(this Shape shape, StreamFloat stream)
        {
            return AddVertexAttribute(shape, ColorAttrInfo, stream);
        }

        public static VtxAttrib AddVertexAttributeNormal(this Shape shape, StreamFloat stream)
        {
            return AddVertexAttribute(shape, NormalAttrInfo, stream);
        }

        public static VtxAttrib AddVertexAttributeTangent(this Shape shape, StreamFloat stream)
        {
            return AddVertexAttribute(shape, TangentAttrInfo, stream);
        }

        public static VtxAttrib AddVertexAttributeBinormal(this Shape shape, StreamFloat stream)
        {
            return AddVertexAttribute(shape, BinormalAttrInfo, stream);
        }

        public static VtxAttrib AddVertexAttributeUv(this Shape shape, StreamFloat stream)
        {
            return AddVertexAttribute(shape, UvAttrInfo, stream);
        }

        public static VtxAttrib AddVertexAttributeBlendWeight(this Shape shape, StreamFloat stream)
        {
            return AddVertexAttribute(shape, BlendWeightAttrInfo, stream);
        }

        public static VtxAttrib AddVertexAttributeBlendIndex(this Shape shape, StreamInt stream)
        {
            return AddVertexAttribute(shape, BlendIndexAttrInfo, stream);
        }

        public static VtxAttrib AddVertexAttributePosition(this Shape shape, IEnumerable<float> streamData)
        {
            return AddVertexAttribute(shape, PositionAttrInfo, streamData);
        }

        public static VtxAttrib AddVertexAttributeColor(this Shape shape, IEnumerable<float> streamData)
        {
            return AddVertexAttribute(shape, ColorAttrInfo, streamData);
        }

        public static VtxAttrib AddVertexAttributeNormal(this Shape shape, IEnumerable<float> streamData)
        {
            return AddVertexAttribute(shape, NormalAttrInfo, streamData);
        }

        public static VtxAttrib AddVertexAttributeTangent(this Shape shape, IEnumerable<float> streamData)
        {
            return AddVertexAttribute(shape, TangentAttrInfo, streamData);
        }

        public static VtxAttrib AddVertexAttributeBinormal(this Shape shape, IEnumerable<float> streamData)
        {
            return AddVertexAttribute(shape, BinormalAttrInfo, streamData);
        }

        public static VtxAttrib AddVertexAttributeUv(this Shape shape, IEnumerable<float> streamData)
        {
            return AddVertexAttribute(shape, UvAttrInfo, streamData);
        }

        public static VtxAttrib AddVertexAttributeBlendWeight(this Shape shape, IEnumerable<float> streamData)
        {
            return AddVertexAttribute(shape, BlendWeightAttrInfo, streamData);
        }

        public static VtxAttrib AddVertexAttributeBlendIndex(this Shape shape, IEnumerable<int> streamData)
        {
            return AddVertexAttribute(shape, BlendIndexAttrInfo, streamData);
        }


        private static int FindUnusedSuffixNumberOfAttributeName(this Shape shape, string name)
        {
            int suffixNumber = 0;
            var attr = shape.ShapeInfo.Vertex.VtxAttribs.LastOrDefault(x => x.Name.StartsWith(name));
            if (attr != null)
            {
                suffixNumber = int.Parse(attr.Name.Last().ToString()) + 1;
            }

            return suffixNumber;
        }

        private static VtxAttrib AddVertexAttribute<TValue>(
            this Shape shape,
            VertexAttrInfo info,
            IEnumerable<TValue> streamData)
        {
            Stream stream = StreamUtility.CreateStream(info.Type, streamData);
            shape.Parent.Streams.Add(stream);
            return shape.AddVertexAttribute(info, stream);
        }

        private static VtxAttrib AddVertexAttribute(
            this Shape shape,
            VertexAttrInfo info,
            Stream stream)
        {
            int suffixNumber = shape.FindUnusedSuffixNumberOfAttributeName(info.NamePrefix);
            return shape.AddVertexAttribute(
                $"{info.NamePrefix}{suffixNumber}",
                $"{info.HintPrefix}{suffixNumber}",
                info.Type,
                stream);
        }
    }
}
