﻿using Nintendo.G3dTool.Entities;
using nw.g3d.nw4f_3dif;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.G3dTool.Extensions
{
    public static class ModelExtensions
    {
        public static void RemoveBone(this Model model, Bone target)
        {
            model.Skeleton.RemoveBone(target);
        }

        public static Material CreateMaterial(this Model model, string materialName)
        {
            var material = new Material()
            {
                Name = materialName
            };
            model.Materials.Add(material);
            return material;
        }

        public static void RemoveMaterial(this Model model, Material target)
        {
            RemoveMaterial(model, target.Name);
        }

        public static void RemoveMaterial(this Model model, string materialName)
        {
            foreach (var shape in model.Shapes.Where(x => x.ShapeInfo.MatName == materialName))
            {
                // とりあえず空にしておく
                shape.ShapeInfo.MatName = string.Empty;
            }

            model.Materials.RemoveAll(x => x.Name == materialName);
        }

        public static Shape CreateShape(
            this Model model,
            Material assignMaterial,
            Bone assignBone,
            IEnumerable<int> indexStreamData)
        {
            StreamInt indexStream = new StreamInt(indexStreamData);
            model.Streams.Add(indexStream);
            return model.CreateShape(assignMaterial, assignBone, indexStream);
        }

        public static Shape CreateShape(
            this Model model,
            Material assignMaterial,
            Bone assignBone,
            StreamInt indexStream)
        {
            var shape = new Shape()
            {
                Name = $"{assignBone.Name}__{assignMaterial.Name}",
            };

            shape.ShapeInfo.MatName = assignMaterial.Name;
            shape.ShapeInfo.OriginalMaterialName = assignMaterial.Name;
            shape.ShapeInfo.OriginalBoneName = assignBone.Name;
            shape.ShapeInfo.Bone = assignBone;

            Vertex vertex = new Vertex();
            model.Vertexes.Add(vertex);
            shape.ShapeInfo.Vertex = vertex;

            shape.Meshes.Add(new Mesh()
            {
                IndexStream = indexStream,
                Mode = mesh_modeType.triangles,
            });

            indexStream.Column = 3;

            model.Shapes.Add(shape);

            return shape;
        }

        /// <summary>
        /// シェイプを取り除きます。
        /// </summary>
        /// <param name="model"></param>
        /// <param name="shape"></param>
        public static void RemoveShape(this Model model, Shape shape)
        {
            model.Shapes.Remove(shape);
            model.RemoveUnreferencedVertexes();
            model.RemoveUnreferencedStreams();
        }

        /// <summary>
        /// 参照されていない頂点データを削除します。
        /// </summary>
        /// <param name="model"></param>
        public static void RemoveUnreferencedVertexes(this Model model)
        {
            var removeTargets = model.Vertexes.Where(x => model.ExaminesVertexReferenceCount(x) == 0).ToArray();
            foreach (var target in removeTargets)
            {
                model.Vertexes.Remove(target);
            }
        }

        /// <summary>
        /// 頂点データの参照数を調べます。
        /// </summary>
        /// <param name="model"></param>
        /// <param name="target"></param>
        /// <returns></returns>
        public static int ExaminesVertexReferenceCount(this Model model, Vertex target)
        {
            int referenceCount = 0;
            foreach (var shape in model.Shapes)
            {
                if (shape.ShapeInfo.Vertex == target)
                {
                    ++referenceCount;
                }
            }

            return referenceCount;
        }

        /// <summary>
        /// モデル内で参照されていないストリームをすべて削除します。
        /// </summary>
        /// <param name="model"></param>
        public static void RemoveUnreferencedStreams(this Model model)
        {
            var removeTargets = model.Streams.Where(x => model.ExaminesStreamReferenceCount(x) == 0).ToArray();
            foreach (var stream in removeTargets)
            {
                model.Streams.Remove(stream);
            }
        }

        /// <summary>
        /// ストリームの参照数を調べます。
        /// </summary>
        /// <param name="model"></param>
        /// <param name="target"></param>
        /// <returns></returns>
        public static int ExaminesStreamReferenceCount(this Model model, Stream target)
        {
            int referenceCount = 0;

            // インデックスストリームからの参照チェック
            if (target.Type == stream_typeType.@int)
            {
                foreach (var shape in model.Shapes)
                {
                    foreach (var mesh in shape.Meshes)
                    {
                        if (mesh.IndexStream == target)
                        {
                            ++referenceCount;
                        }
                    }
                }
            }

            // 頂点属性からの参照チェック
            foreach (var vertex in model.Vertexes)
            {
                foreach (var attr in vertex.VtxAttribs)
                {
                    if (attr.Stream == target)
                    {
                        ++referenceCount;
                    }
                }
            }

            // ユーザーデータからの参照チェック
            if (target.Type == stream_typeType.@byte)
            {
                foreach (var userData in model.UserDatas)
                {
                    if (userData.Type != UserDataType.Stream)
                    {
                        continue;
                    }

                    var userDataStream = userData as UserDataStream;
                    if (userDataStream.Stream == target)
                    {
                        ++referenceCount;
                    }
                }
            }

            return referenceCount;
        }

        /// <summary>
        /// モデル内で参照されていないマテリアルを削除します。
        /// </summary>
        /// <param name="model"></param>
        public static void RemoveUnreferencedMaterials(this Model model)
        {
            var removeTargets = model.Materials.Where(x => model.ExaminesMaterialReferenceCount(x) == 0).ToArray();
            foreach (var target in removeTargets)
            {
                model.Materials.Remove(target);
            }
        }

        /// <summary>
        /// モデル内のマテリアルの参照数を調べます。
        /// </summary>
        /// <param name="model"></param>
        /// <param name="target"></param>
        /// <returns></returns>
        public static int ExaminesMaterialReferenceCount(this Model model, Material target)
        {
            int referenceCount = 0;
            foreach (var shape in model.Shapes)
            {
                if (shape.ShapeInfo.MatName == target.Name)
                {
                    ++referenceCount;
                }
            }

            return referenceCount;
        }
    }
}
