﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class VtxBuffer : ObservableEntity<vtx_bufferType>, IDeepCopyable<VtxBuffer>, IDeepCopyFrom<VtxBuffer>
    {
        private readonly ObservableList<Input> @inputs = new ObservableList<Input>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public VtxBuffer()
        {
            this.@inputs.CollectionChanged += OnCollectionChanged<Input>;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public VtxBuffer(VtxBuffer source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public VtxBuffer(vtx_bufferType source)
        {
            this.@inputs.CollectionChanged += OnCollectionChanged<Input>;
            if (source.@input_array != null)
            {
                this.@inputs.Clear();
                foreach (var elem in source.@input_array.Items)
                {
                    this.@inputs.Add(new Input(elem));
                }
            }
        }

        public ObservableList<Input> Inputs
        {
            get
            {
                return this.@inputs;
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override vtx_bufferType CreateSerializableData()
        {
            var writeData = new vtx_bufferType();
            if (this.@Inputs.Count > 0)
            {
                writeData.@input_array = new input_arrayType();
                writeData.@input_array.Items = this.@inputs.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@input_array.Items.Length == 0)
                {
                    writeData.@input_array = null;
                }
            }
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        VtxBuffer IDeepCopyable<VtxBuffer>.DeepCopy()
        {
            return new VtxBuffer(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(VtxBuffer source)
        {
            CopyElements(source.@inputs, this.@inputs);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.Inputs)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@inputs)
            {
                elem.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@inputs)
            {
                elem.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@inputs)
            {
                elem.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
