﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class VtxAttrib : ObservableEntity<vtx_attribType>, IChildEntity, IStreamIndex, IDeepCopyable<VtxAttrib>, IDeepCopyFrom<VtxAttrib>
    {
        private string @name = string.Empty;
        private string @hint = string.Empty;
        private vtx_attrib_typeType @type;
        private vtx_attrib_quantize_typeType @quantizeType;
        private int @count;
        private int @streamIndex;
        private Vertex parent = null;
        private Stream @stream = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public VtxAttrib()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public VtxAttrib(VtxAttrib source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public VtxAttrib(vtx_attribType source)
        {
            this.@name = source.@name;
            this.@hint = source.@hint;
            this.@type = source.@type;
            this.@quantizeType = source.@quantize_type;
            this.@count = source.@count;
            this.@streamIndex = source.@stream_index;
        }

        public string Name
        {
            get
            {
                return this.@name;
            }

            set
            {
                if (this.@name == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@name, value, () => this.CalcCRC());
            }
        }

        public string Hint
        {
            get
            {
                return this.@hint;
            }

            set
            {
                if (this.@hint == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@hint, value, () => this.CalcCRC());
            }
        }

        public vtx_attrib_typeType Type
        {
            get
            {
                return this.@type;
            }

            set
            {
                if (this.@type == value)
                {
                    return;
                }

                this.SetProperty(ref this.@type, value, () => this.CalcCRC());
            }
        }

        public vtx_attrib_quantize_typeType QuantizeType
        {
            get
            {
                return this.@quantizeType;
            }

            set
            {
                if (this.@quantizeType == value)
                {
                    return;
                }

                this.SetProperty(ref this.@quantizeType, value, () => this.CalcCRC());
            }
        }

        public int Count
        {
            get
            {
                if (this.Stream != null)
                {
                    return this.Stream.Count / EstimateElementCountPerVertex(this.Type);
                }
                return this.@count;
            }
        }

        int IStreamIndex.StreamIndex
        {
            get
            {
                if (this.Stream != null)
                {
                    // 親が見つからない場合はインデックスを求められないので -1 を返す
                    IRootEntity rootEntity = Utility.FindRootEntity((this as IChildEntity).Parent);
                    if (rootEntity == null)
                    {
                        return -1;
                    }

                    var sourceProp = rootEntity.GetType().GetProperty("Streams").GetValue(rootEntity) as ObservableList<Stream>;
                    return sourceProp.IndexOf(this.Stream);
                }
                return this.@streamIndex;
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is Vertex : true);
                this.SetProperty(ref this.parent, value as Vertex, () => this.CalcCRC());
            }
        }

        public Vertex Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(Vertex);
            }
        }

        public Stream Stream
        {
            get
            {
                return this.@stream;
            }

            set
            {
                if (this.@stream == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@stream, value, () => this.CalcCRC());
            }
        }

        internal static int EstimateElementCountPerVertex(vtx_attrib_typeType type)
        {
            char typeNameLastChar = type.ToString().Last();
            if (char.IsDigit(typeNameLastChar))
            {
                return int.Parse(typeNameLastChar.ToString());
            }
            else
            {
                return 1;
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override vtx_attribType CreateSerializableData()
        {
            var writeData = new vtx_attribType();
            writeData.@name = this.Name;
            writeData.@hint = this.Hint;
            writeData.@type = this.Type;
            writeData.@quantize_type = this.QuantizeType;
            writeData.@count = this.Count;
            {
                IRootEntity rootEntity = Utility.FindRootEntity((this as IChildEntity).Parent);
                Ensure.Operation.NotNull(rootEntity);
                var sourceProp = rootEntity.GetType().GetProperty("Streams").GetValue(rootEntity) as ObservableList<Stream>;
                // 読み取り高速化のために不要なデータを読み飛ばす場合があるので、参照先が存在しない場合を許容します
                Ensure.Operation.True((sourceProp.Count == 0) || (Stream != null));
                writeData.stream_index = (this as IStreamIndex).StreamIndex;
            }
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        VtxAttrib IDeepCopyable<VtxAttrib>.DeepCopy()
        {
            return new VtxAttrib(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(VtxAttrib source)
        {
            this.@name = source.@name;
            this.@hint = source.@hint;
            this.@type = source.@type;
            this.@quantizeType = source.@quantizeType;
            this.@count = source.@count;
            this.@streamIndex = source.@streamIndex;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Name)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Hint)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Type)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.QuantizeType)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Count)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32((this as IStreamIndex).StreamIndex)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
