﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class VertexShapeAnim : ObservableEntity<vertex_shape_animType>, IChildEntity, IDeepCopyable<VertexShapeAnim>, IDeepCopyFrom<VertexShapeAnim>
    {
        private readonly ObservableList<ShapeAnimTarget> @shapeAnimTargets = new ObservableList<ShapeAnimTarget>();
        private string @shapeName = string.Empty;
        private string @baseName = string.Empty;
        private ShapeAnim parent = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public VertexShapeAnim()
        {
            this.@shapeAnimTargets.CollectionChanged += OnCollectionChanged<ShapeAnimTarget>;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public VertexShapeAnim(VertexShapeAnim source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public VertexShapeAnim(vertex_shape_animType source)
        {
            this.@shapeAnimTargets.CollectionChanged += OnCollectionChanged<ShapeAnimTarget>;
            if (source.@shape_anim_target != null)
            {
                this.@shapeAnimTargets.Clear();
                foreach (var elem in source.@shape_anim_target)
                {
                    this.@shapeAnimTargets.Add(new ShapeAnimTarget(elem));
                }
            }
            this.@shapeName = source.@shape_name;
            this.@baseName = source.@base_name;
        }

        public ObservableList<ShapeAnimTarget> ShapeAnimTargets
        {
            get
            {
                return this.@shapeAnimTargets;
            }
        }

        public string ShapeName
        {
            get
            {
                return this.@shapeName;
            }

            set
            {
                if (this.@shapeName == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@shapeName, value, () => this.CalcCRC());
            }
        }

        public string BaseName
        {
            get
            {
                return this.@baseName;
            }

            set
            {
                if (this.@baseName == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@baseName, value, () => this.CalcCRC());
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is ShapeAnim : true);
                this.SetProperty(ref this.parent, value as ShapeAnim, () => this.CalcCRC());
            }
        }

        public ShapeAnim Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(ShapeAnim);
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override vertex_shape_animType CreateSerializableData()
        {
            Ensure.Operation.Range(this.ShapeAnimTargets.Count, 1, int.MaxValue);
            var writeData = new vertex_shape_animType();
            if (this.@ShapeAnimTargets.Count > 0)
            {
                writeData.@shape_anim_target = this.@shapeAnimTargets.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@shape_anim_target.Length == 0)
                {
                    writeData.@shape_anim_target = null;
                }
            }
            writeData.@shape_name = this.ShapeName;
            writeData.@base_name = this.BaseName;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        VertexShapeAnim IDeepCopyable<VertexShapeAnim>.DeepCopy()
        {
            return new VertexShapeAnim(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(VertexShapeAnim source)
        {
            CopyElements(source.@shapeAnimTargets, this.@shapeAnimTargets);
            this.@shapeName = source.@shapeName;
            this.@baseName = source.@baseName;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.ShapeAnimTargets)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ShapeName)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.BaseName)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@shapeAnimTargets)
            {
                elem.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@shapeAnimTargets)
            {
                elem.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@shapeAnimTargets)
            {
                elem.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
