﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class UserDataInt : UserData, IDeepCopyable<UserDataInt>, IDeepCopyFrom<UserDataInt>
    {
        private readonly ObservableList<int> @values = new ObservableList<int>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UserDataInt()
        {
            this.@values.CollectionChanged += OnValueCollectionChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public UserDataInt(UserDataInt source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public UserDataInt(user_dataType source)
            : base(source)
        {
            this.@values.CollectionChanged += OnValueCollectionChanged;
            {
                var valueArray = G3dDataParser.ParseIntArray((source.Item as user_intType).Value);
                if (valueArray != null)
                {
                    this.@values.Add(valueArray);
                }
            }
        }

        public ObservableList<int> Values
        {
            get
            {
                return this.@values;
            }
        }

        public override UserDataType Type
        {
            get
            {
                return UserDataType.Int;
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override user_dataType CreateSerializableData()
        {
            var writeData = base.CreateSerializableData();
            if (writeData.Item == null)
            {
                writeData.Item = new user_intType();
            }
            (writeData.Item as user_intType).count = this.Values.Count;
            (writeData.Item as user_intType).Value = G3dDataFormatter.FormatStreamData(this.Values.ToArray(), this.Values.Count);
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        UserDataInt IDeepCopyable<UserDataInt>.DeepCopy()
        {
            return new UserDataInt(this);
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        private protected override UserData DeepCopy()
        {
            return new UserDataInt(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(UserDataInt source)
        {
            base.DeepCopyFrom(source);
            CopyValues(source.@values, this.@values);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(base.CreateCRCInternal()));
            foreach (var elem in this.Values)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(elem)));
            }
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
