﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using nw.g3d.nw4f_3dif;
using Opal.Utilities;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;

namespace Nintendo.G3dTool.Entities
{
    public class ToolDataElement : ObservableEntity<XmlElement>, IDeepCopyable<ToolDataElement>
    {
        private XmlElement unkownToolData;

        public string Id { get; set; } = string.Empty;

        /// <summary>
        /// ツールデータアイテムを取得します。
        /// </summary>
        public IToolDataObject ToolDataObject { get; private set; }

        public ToolDataElement(XmlElement source)
        {
            this.ToolDataObject = IfDomUtility.ConvertXmlElementToObject(source) as IToolDataObject;
            if (this.ToolDataObject == null)
            {
                this.unkownToolData = source;
            }
            else
            {
                this.ToolDataObject.PropertyChanged += ToolDataObject_PropertyChanged;
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="obj">実データ</param>
        public ToolDataElement(IToolDataObject obj)
        {
            this.ToolDataObject = obj;
            this.ToolDataObject.PropertyChanged += ToolDataObject_PropertyChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="obj"></param>
        public ToolDataElement(ToolDataElement source)
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override XmlElement CreateSerializableData()
        {
            if (this.ToolDataObject != null)
            {
                object serializableData = this.ToolDataObject.CreateSerializableData();
                if (serializableData == null)
                {
                    return null;
                }

                return IfDomUtility.ConvertObjectToXmlElement(this.ToolDataObject.CreateSerializableData());
            }

            return this.unkownToolData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        ToolDataElement IDeepCopyable<ToolDataElement>.DeepCopy()
        {
            return new ToolDataElement(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(ToolDataElement source)
        {
            if (source.ToolDataObject != null)
            {
                this.ToolDataObject = source.ToolDataObject.DeepCopy();
            }
            else
            {
                this.unkownToolData = source.unkownToolData.Clone() as XmlElement;
            }
        }

        /// <summary>
        /// エンティティのCRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRCの値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            // 不明なToolDataの場合はどうせ編集できないので0返す。
            return this.ToolDataObject?.CalculateCrc() ?? 0;
        }

        private void ToolDataObject_PropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            this.CalcCRC();
        }
    }
}
