﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class TexPattern : ObservableEntity<tex_patternType>, IDeepCopyable<TexPattern>, IDeepCopyFrom<TexPattern>
    {
        private int @patternIndex;
        private string @texName = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public TexPattern()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public TexPattern(TexPattern source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public TexPattern(tex_patternType source)
        {
            this.@patternIndex = source.@pattern_index;
            this.@texName = source.@tex_name;
        }

        public int PatternIndex
        {
            get
            {
                return this.@patternIndex;
            }

            set
            {
                if (this.@patternIndex == value)
                {
                    return;
                }

                this.SetProperty(ref this.@patternIndex, value, () => this.CalcCRC());
            }
        }

        public string TexName
        {
            get
            {
                return this.@texName;
            }

            set
            {
                if (this.@texName == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@texName, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override tex_patternType CreateSerializableData()
        {
            var writeData = new tex_patternType();
            writeData.@pattern_index = this.PatternIndex;
            writeData.@tex_name = this.TexName;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        TexPattern IDeepCopyable<TexPattern>.DeepCopy()
        {
            return new TexPattern(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(TexPattern source)
        {
            this.@patternIndex = source.@patternIndex;
            this.@texName = source.@texName;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.PatternIndex)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.TexName)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
