﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Submesh : ObservableEntity<submeshType>, IDeepCopyable<Submesh>, IDeepCopyFrom<Submesh>
    {
        private int @offset;
        private int @count;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Submesh()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Submesh(Submesh source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Submesh(submeshType source)
        {
            this.@offset = source.@offset;
            this.@count = source.@count;
        }

        public int Offset
        {
            get
            {
                return this.@offset;
            }

            set
            {
                if (this.@offset == value)
                {
                    return;
                }

                this.SetProperty(ref this.@offset, value, () => this.CalcCRC());
            }
        }

        public int Count
        {
            get
            {
                return this.@count;
            }

            set
            {
                if (this.@count == value)
                {
                    return;
                }

                this.SetProperty(ref this.@count, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override submeshType CreateSerializableData()
        {
            var writeData = new submeshType();
            writeData.@offset = this.Offset;
            writeData.@count = this.Count;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Submesh IDeepCopyable<Submesh>.DeepCopy()
        {
            return new Submesh(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Submesh source)
        {
            this.@offset = source.@offset;
            this.@count = source.@count;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Offset)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Count)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
