﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Skeleton : ObservableEntity<skeletonType>, IChildEntity, IDeepCopyable<Skeleton>, IDeepCopyFrom<Skeleton>, IDisposable
    {
        private readonly SkeletonInfo @skeletonInfo = new SkeletonInfo();
        private Model parent = null;
        private Bone @rootBone = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Skeleton()
        {
            this.@skeletonInfo.PropertyChanged += this.OnPropertyChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Skeleton(Skeleton source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Skeleton(skeletonType source)
        {
            if (source.@skeleton_info != null)
            {
                this.@skeletonInfo = new SkeletonInfo(source.@skeleton_info);
            }
            else
            {
                this.@skeletonInfo = new SkeletonInfo();
            }
            if (source.@bone_array != null)
            {
                List<Bone> bones = new List<Bone>();
                foreach (var elem in source.@bone_array.Items)
                {
                    bones.Add(new Bone(elem));
                }
                this.RootBone = bones.FirstOrDefault(x => string.IsNullOrEmpty(x.ParentName));

                // 親ボーンの参照設定
                foreach (var bone in bones)
                {
                    if (string.IsNullOrEmpty(bone.ParentName))
                    {
                        continue;
                    }

                    var parentBone = bones.First(x => x.Name == bone.ParentName);
                    parentBone.ChildBones.Add(bone);
                }
            }
            this.@skeletonInfo.PropertyChanged += this.OnPropertyChanged;
        }

        public void Dispose()
        {
            if (this.@rootBone != null)
            {
                this.@rootBone.PropertyChanged -= this.OnPropertyChanged;
            }
        }

        public SkeletonInfo SkeletonInfo
        {
            get
            {
                return this.@skeletonInfo;
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is Model : true);
                this.SetProperty(ref this.parent, value as Model, () => this.CalcCRC());
            }
        }

        public Model Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(Model);
            }
        }

        public Bone RootBone
        {
            get
            {
                return this.@rootBone;
            }

            set
            {
                if (this.@rootBone == value)
                {
                    return;
                }

                var oldRootBone = this.rootBone;
                if (this.@rootBone != null)
                {
                    this.@rootBone.PropertyChanged -= this.OnPropertyChanged;
                }

                if (value != null)
                {
                    value.PropertyChanged += this.OnPropertyChanged;
                }

                this.SetProperty(ref this.@rootBone, value, () => this.CalcCRC());
                if (oldRootBone != null)
                {
                    (oldRootBone as IChildEntity).Parent = null;
                    oldRootBone.UpdateShapeReferences();
                }
                if (value != null)
                {
                    (value as IChildEntity).Parent = this;
                    value.UpdateShapeReferences();
                }
            }
        }

        public int CountBones()
        {
            return CountBones(this.RootBone, new List<Bone>());
        }

        private int CountBones(Bone rootBone, List<Bone> circularRefCheckList)
        {
            Ensure.Argument.True(!circularRefCheckList.Contains(rootBone), $"Circular reference was found. Error bone name is {rootBone.Name}.");
            circularRefCheckList.Add(rootBone);
            int count = 1;
            foreach (var childBone in rootBone.ChildBones)
            {
                count += CountBones(childBone, circularRefCheckList);
            }

            return count;
        }

        public IEnumerable<Bone> EnumerateBones()
        {
            return EnumerateBones(this.RootBone, new List<Bone>());
        }

        private IEnumerable<Bone> EnumerateBones(Bone rootBone, List<Bone> circularRefCheckList)
        {
            Ensure.Argument.True(!circularRefCheckList.Contains(rootBone), $"Circular reference was found. Error bone name is {rootBone.Name}.");
            yield return rootBone;
            circularRefCheckList.Add(rootBone);
            foreach (var childBone in rootBone.ChildBones)
            {
                foreach (var enumeratedChildBone in EnumerateBones(childBone, circularRefCheckList))
                {
                    yield return enumeratedChildBone;
                }
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override skeletonType CreateSerializableData()
        {
            var writeData = new skeletonType();
            writeData.@skeleton_info = this.@skeletonInfo.CreateSerializableData() as skeletonInfoType;
            {
                if (this.CountBones() > 0)
                {
                    writeData.bone_array = new bone_arrayType();
                    writeData.bone_array.bone = this.EnumerateBones().Select(x => x.CreateSerializableData()).ToArray();
                }
            }
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Skeleton IDeepCopyable<Skeleton>.DeepCopy()
        {
            return new Skeleton(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Skeleton source)
        {
            this.@skeletonInfo.DeepCopyFrom(source.@skeletonInfo);
            this.RootBone = new Bone(source.RootBone);
            foreach (var bone in this.EnumerateBones())
            {
                Ensure.Operation.True((bone.ParentBone != null) || (bone.Parent != null), $"ParentBone or Parent must not be null. {bone.Name}");
            }
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.SkeletonInfo.HashValue)));
            if (this.RootBone != null)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.RootBone.HashValue)));
            }
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            this.@skeletonInfo.AutoCalc = this.AutoCalc;
            if (this.@rootBone != null)
            {
                this.@rootBone.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            this.@skeletonInfo.Reset();
            if (this.@rootBone != null)
            {
                this.@rootBone.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            this.@skeletonInfo.Refresh();
            if (this.@rootBone != null)
            {
                this.@rootBone.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
