﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Shape : ObservableEntity<shapeType>, IChildEntity, IDeepCopyable<Shape>, IDeepCopyFrom<Shape>
    {
        private readonly ShapeInfo @shapeInfo = new ShapeInfo();
        private readonly ObservableList<Mesh> @meshes = new ObservableList<Mesh>();
        private readonly ObservableList<KeyShape> @keyShapes = new ObservableList<KeyShape>();
        private string @name = string.Empty;
        private Model parent = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Shape()
        {
            this.@shapeInfo.PropertyChanged += this.OnPropertyChanged;
            this.@meshes.CollectionChanged += OnCollectionChanged<Mesh>;
            this.@keyShapes.CollectionChanged += OnCollectionChanged<KeyShape>;
            (this.@shapeInfo as IChildEntity).Parent = this;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Shape(Shape source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Shape(shapeType source)
        {
            this.@meshes.CollectionChanged += OnCollectionChanged<Mesh>;
            this.@keyShapes.CollectionChanged += OnCollectionChanged<KeyShape>;
            if (source.@shape_info != null)
            {
                this.@shapeInfo = new ShapeInfo(source.@shape_info);
            }
            else
            {
                this.@shapeInfo = new ShapeInfo();
            }
            (this.@shapeInfo as IChildEntity).Parent = this;
            if (source.@mesh_array != null)
            {
                this.@meshes.Clear();
                foreach (var elem in source.@mesh_array.Items)
                {
                    this.@meshes.Add(new Mesh(elem));
                }
            }
            if (source.@key_shape_array != null)
            {
                this.@keyShapes.Clear();
                foreach (var elem in source.@key_shape_array.Items)
                {
                    this.@keyShapes.Add(new KeyShape(elem));
                }
            }
            this.@name = source.@name;
            this.@shapeInfo.PropertyChanged += this.OnPropertyChanged;
        }

        public ShapeInfo ShapeInfo
        {
            get
            {
                return this.@shapeInfo;
            }
        }

        public ObservableList<Mesh> Meshes
        {
            get
            {
                return this.@meshes;
            }
        }

        public ObservableList<KeyShape> KeyShapes
        {
            get
            {
                return this.@keyShapes;
            }
        }

        public string Name
        {
            get
            {
                return this.@name;
            }

            set
            {
                if (this.@name == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@name, value, () => this.CalcCRC());
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is Model : true);
                Model oldModel = this.Parent;

                this.SetProperty(ref this.parent, value as Model, () => this.CalcCRC());

                // 所属する親が変化したのでボーンのシェイプ関連付け直し
                if (oldModel != null)
                {
                    var bone = oldModel.Skeleton.EnumerateBones().FirstOrDefault(x => x.Shapes.Contains(this));
                    if (bone != null)
                    {
                        bone.SourceShapes.Remove(this);
                    }
                }
                if (this.Parent != null)
                {
                    var bone = this.Parent.Skeleton.EnumerateBones().FirstOrDefault(x => x.Name == this.ShapeInfo.BoneName);
                    if (bone != null)
                    {
                        bone.SourceShapes.Add(this);
                        this.ShapeInfo.Bone = bone;
                    }
                }
                else
                {
                    this.ShapeInfo.Bone = null;
                }
            }
        }

        public Model Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(Model);
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override shapeType CreateSerializableData()
        {
            var writeData = new shapeType();
            writeData.@shape_info = this.@shapeInfo.CreateSerializableData() as shape_infoType;
            if (this.@Meshes.Count > 0)
            {
                writeData.@mesh_array = new mesh_arrayType();
                writeData.@mesh_array.Items = this.@meshes.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@mesh_array.Items.Length == 0)
                {
                    writeData.@mesh_array = null;
                }
            }
            if (this.@KeyShapes.Count > 0)
            {
                writeData.@key_shape_array = new key_shape_arrayType();
                writeData.@key_shape_array.Items = this.@keyShapes.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@key_shape_array.Items.Length == 0)
                {
                    writeData.@key_shape_array = null;
                }
            }
            writeData.@name = this.Name;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Shape IDeepCopyable<Shape>.DeepCopy()
        {
            return new Shape(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Shape source)
        {
            this.@shapeInfo.DeepCopyFrom(source.@shapeInfo);
            CopyElements(source.@meshes, this.@meshes);
            CopyElements(source.@keyShapes, this.@keyShapes);
            this.@name = source.@name;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ShapeInfo.HashValue)));
            foreach (var elem in this.Meshes)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.KeyShapes)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Name)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            this.@shapeInfo.AutoCalc = this.AutoCalc;
            foreach (var elem in this.@meshes)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@keyShapes)
            {
                elem.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            this.@shapeInfo.Reset();
            foreach (var elem in this.@meshes)
            {
                elem.Reset();
            }
            foreach (var elem in this.@keyShapes)
            {
                elem.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            this.@shapeInfo.Refresh();
            foreach (var elem in this.@meshes)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@keyShapes)
            {
                elem.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
