﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class ShaderVariation : RootEntity<shader_variationType>, IDeepCopyable<ShaderVariation>, IDeepCopyFrom<ShaderVariation>, IToolData
    {
        private readonly ShaderVariationInfo @shaderVariationInfo = new ShaderVariationInfo();
        private readonly ObservableList<TargetShader> @targetShaders = new ObservableList<TargetShader>();
        private readonly ObservableList<Stream> @streams = new ObservableList<Stream>();
        private readonly ToolData @toolData = new ToolData();
        private readonly UserToolData @userToolData = new UserToolData();
        private shader_variation_versionType @version;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        internal ShaderVariation()
        {
            this.@shaderVariationInfo.PropertyChanged += this.OnPropertyChanged;
            this.@targetShaders.CollectionChanged += OnCollectionChanged<TargetShader>;
            this.@streams.CollectionChanged += OnCollectionChanged<Stream>;
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShaderVariation(ShaderVariation source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShaderVariation(shader_variationType source)
        {
            this.@targetShaders.CollectionChanged += OnCollectionChanged<TargetShader>;
            this.@streams.CollectionChanged += OnCollectionChanged<Stream>;
            if (source.@shader_variation_info != null)
            {
                this.@shaderVariationInfo = new ShaderVariationInfo(source.@shader_variation_info);
            }
            else
            {
                this.@shaderVariationInfo = new ShaderVariationInfo();
            }
            if (source.@target_shader_array != null)
            {
                this.@targetShaders.Clear();
                foreach (var elem in source.@target_shader_array.Items)
                {
                    this.@targetShaders.Add(new TargetShader(elem));
                }
            }
            if (source.@stream_array != null)
            {
                this.@streams.Clear();
                foreach (var elem in source.@stream_array.Items)
                {
                    this.@streams.Add(DataModelConverter.Convert(elem));
                }
            }
            if (source.@tool_data != null)
            {
                this.@toolData = new ToolData(source.@tool_data);
            }
            else
            {
                this.@toolData = new ToolData();
            }
            if (source.@user_tool_data != null)
            {
                this.@userToolData = new UserToolData(source.@user_tool_data);
            }
            else
            {
                this.@userToolData = new UserToolData();
            }
            this.@version = source.@version;
            this.@shaderVariationInfo.PropertyChanged += this.OnPropertyChanged;
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
            this.ResolveInternalReferences();
        }

        public ShaderVariationInfo ShaderVariationInfo
        {
            get
            {
                return this.@shaderVariationInfo;
            }
        }

        public ObservableList<TargetShader> TargetShaders
        {
            get
            {
                return this.@targetShaders;
            }
        }

        public override ObservableList<Stream> Streams
        {
            get
            {
                return this.@streams;
            }
        }

        public ToolData ToolData
        {
            get
            {
                return this.@toolData;
            }
        }

        public UserToolData UserToolData
        {
            get
            {
                return this.@userToolData;
            }
        }

        public shader_variation_versionType Version
        {
            get
            {
                return this.@version;
            }

            set
            {
                if (this.@version == value)
                {
                    return;
                }

                this.SetProperty(ref this.@version, value, () => this.CalcCRC());
            }
        }

        public override void ResolveInternalReferences()
        {
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override shader_variationType CreateSerializableData()
        {
            Ensure.Operation.Range(this.TargetShaders.Count, 1, int.MaxValue);
            var writeData = new shader_variationType();
            writeData.@shader_variation_info = this.@shaderVariationInfo.CreateSerializableData() as shader_variation_infoType;
            if (this.@TargetShaders.Count > 0)
            {
                writeData.@target_shader_array = new target_shader_arrayType();
                writeData.@target_shader_array.Items = this.@targetShaders.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@target_shader_array.Items.Length == 0)
                {
                    writeData.@target_shader_array = null;
                }
            }
            if (this.IsStreamSerializationEnabled)
            {
                if (this.@Streams.Count > 0)
                {
                    writeData.@stream_array = new stream_arrayType();
                    writeData.@stream_array.Items = this.@streams.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                    if (writeData.@stream_array.Items.Length == 0)
                    {
                        writeData.@stream_array = null;
                    }
                }
            }
            writeData.@tool_data = this.@toolData.CreateSerializableData() as tool_dataType;
            writeData.@user_tool_data = this.@userToolData.CreateSerializableData() as user_tool_dataType;
            writeData.@version = this.Version;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        ShaderVariation IDeepCopyable<ShaderVariation>.DeepCopy()
        {
            return new ShaderVariation(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(ShaderVariation source)
        {
            this.@shaderVariationInfo.DeepCopyFrom(source.@shaderVariationInfo);
            CopyElements(source.@targetShaders, this.@targetShaders);
            CopyElements(source.@streams, this.@streams);
            this.@toolData.DeepCopyFrom(source.@toolData);
            this.@userToolData.DeepCopyFrom(source.@userToolData);
            this.@version = source.@version;
            this.ResolveInternalReferences();
        }

        private protected override IRootEntity DeepCopy()
        {
            return new ShaderVariation(this);
        }

        private protected override void DeepCopyFrom(IRootEntity source)
        {
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.True(source is ShaderVariation);
            this.DeepCopyFrom(source as ShaderVariation);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ShaderVariationInfo.HashValue)));
            foreach (var elem in this.TargetShaders)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.Streams)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ToolData.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.UserToolData.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Version)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            this.@shaderVariationInfo.AutoCalc = this.AutoCalc;
            foreach (var elem in this.@targetShaders)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@streams)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            this.@toolData.AutoCalc = this.AutoCalc;
            this.@userToolData.AutoCalc = this.AutoCalc;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            this.@shaderVariationInfo.Reset();
            foreach (var elem in this.@targetShaders)
            {
                elem.Reset();
            }
            foreach (var elem in this.@streams)
            {
                elem.Reset();
            }
            this.@toolData.Reset();
            this.@userToolData.Reset();
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            this.@shaderVariationInfo.Refresh();
            foreach (var elem in this.@targetShaders)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@streams)
            {
                elem.Refresh();
            }
            this.@toolData.Refresh();
            this.@userToolData.Refresh();
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
