﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class ShaderParamBool4 : ShaderParam, IDeepCopyable<ShaderParamBool4>, IDeepCopyFrom<ShaderParamBool4>
    {
        private readonly Bool4 @value = new Bool4();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShaderParamBool4()
        {
            this.@value.PropertyChanged += this.OnPropertyChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShaderParamBool4(ShaderParamBool4 source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShaderParamBool4(shader_paramType source)
            : base(source)
        {
            Ensure.Argument.AreEqual(shader_param_typeType.@bool4, source.type);
            this.@value = Bool4.Parse(source.Value);
            this.@value.PropertyChanged += this.OnPropertyChanged;
        }

        public ShaderParamBool4(UniformVar definition)
            : this()
        {
            this.Id = definition.Id;
            this.SetToDefault(definition);
        }

        public override shader_param_typeType Type
        {
            get
            {
                return shader_param_typeType.@bool4;
            }
        }

        public Bool4 Value
        {
            get
            {
                return this.@value;
            }
        }

        private protected override void SetToDefaultInternal(UniformVar definition)
        {
            SetValues(definition.Default, this.Value.Values);
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override shader_paramType CreateSerializableData()
        {
            var writeData = base.CreateSerializableData();
            writeData.@type = this.Type;
            writeData.Value = this.@value.ToString().ToLower();
            writeData.type = shader_param_typeType.@bool4;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        ShaderParamBool4 IDeepCopyable<ShaderParamBool4>.DeepCopy()
        {
            return new ShaderParamBool4(this);
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        private protected override ShaderParam DeepCopy()
        {
            return new ShaderParamBool4(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(ShaderParamBool4 source)
        {
            base.DeepCopyFrom(source);
            this.@value.DeepCopyFrom(source.@value);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(base.CreateCRCInternal()));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Value.HashValue)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            base.SetAutoCalcFlagInternal();
            this.@value.AutoCalc = this.AutoCalc;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            base.ResetInternal();
            this.@value.Reset();
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            base.RefreshInternal();
            this.@value.Refresh();
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
