﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class ShaderDefinitionInfo : ObservableEntity<shader_definition_infoType>, IDeepCopyable<ShaderDefinitionInfo>, IDeepCopyFrom<ShaderDefinitionInfo>
    {
        private int @codePage;
        private string @configPath = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShaderDefinitionInfo()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShaderDefinitionInfo(ShaderDefinitionInfo source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShaderDefinitionInfo(shader_definition_infoType source)
        {
            this.@codePage = source.@code_page;
            this.@configPath = source.@config_path;
        }

        public int CodePage
        {
            get
            {
                return this.@codePage;
            }

            set
            {
                if (this.@codePage == value)
                {
                    return;
                }

                this.SetProperty(ref this.@codePage, value, () => this.CalcCRC());
            }
        }

        public string ConfigPath
        {
            get
            {
                return this.@configPath;
            }

            set
            {
                if (this.@configPath == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@configPath, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override shader_definition_infoType CreateSerializableData()
        {
            var writeData = new shader_definition_infoType();
            writeData.@code_page = this.CodePage;
            writeData.@config_path = this.ConfigPath;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        ShaderDefinitionInfo IDeepCopyable<ShaderDefinitionInfo>.DeepCopy()
        {
            return new ShaderDefinitionInfo(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(ShaderDefinitionInfo source)
        {
            this.@codePage = source.@codePage;
            this.@configPath = source.@configPath;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.CodePage)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ConfigPath)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
