﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Shader : ObservableEntity<shaderType>, IDeepCopyable<Shader>, IDeepCopyFrom<Shader>
    {
        private readonly VertexShader @vertexShader = new VertexShader();
        private readonly GeometryShader @geometryShader = new GeometryShader();
        private readonly FragmentShader @fragmentShader = new FragmentShader();
        private readonly ComputeShader @computeShader = new ComputeShader();
        private readonly ObservableList<Macro> @macros = new ObservableList<Macro>();
        private readonly ObservableList<Variation> @variations = new ObservableList<Variation>();
        private string @name = string.Empty;
        private bool @materialShader;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Shader()
        {
            this.@vertexShader.PropertyChanged += this.OnPropertyChanged;
            this.@geometryShader.PropertyChanged += this.OnPropertyChanged;
            this.@fragmentShader.PropertyChanged += this.OnPropertyChanged;
            this.@computeShader.PropertyChanged += this.OnPropertyChanged;
            this.@macros.CollectionChanged += OnCollectionChanged<Macro>;
            this.@variations.CollectionChanged += OnCollectionChanged<Variation>;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Shader(Shader source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Shader(shaderType source)
        {
            this.@macros.CollectionChanged += OnCollectionChanged<Macro>;
            this.@variations.CollectionChanged += OnCollectionChanged<Variation>;
            if (source.@vertex_shader != null)
            {
                this.@vertexShader = new VertexShader(source.@vertex_shader);
            }
            else
            {
                this.@vertexShader = new VertexShader();
            }
            if (source.@geometry_shader != null)
            {
                this.@geometryShader = new GeometryShader(source.@geometry_shader);
            }
            else
            {
                this.@geometryShader = new GeometryShader();
            }
            if (source.@fragment_shader != null)
            {
                this.@fragmentShader = new FragmentShader(source.@fragment_shader);
            }
            else
            {
                this.@fragmentShader = new FragmentShader();
            }
            if (source.@compute_shader != null)
            {
                this.@computeShader = new ComputeShader(source.@compute_shader);
            }
            else
            {
                this.@computeShader = new ComputeShader();
            }
            if (source.@macro_array != null)
            {
                this.@macros.Clear();
                foreach (var elem in source.@macro_array.Items)
                {
                    this.@macros.Add(new Macro(elem));
                }
            }
            if (source.@variation_array != null)
            {
                this.@variations.Clear();
                foreach (var elem in source.@variation_array.Items)
                {
                    this.@variations.Add(new Variation(elem));
                }
            }
            this.@name = source.@name;
            this.@materialShader = source.@material_shader;
            this.@vertexShader.PropertyChanged += this.OnPropertyChanged;
            this.@geometryShader.PropertyChanged += this.OnPropertyChanged;
            this.@fragmentShader.PropertyChanged += this.OnPropertyChanged;
            this.@computeShader.PropertyChanged += this.OnPropertyChanged;
        }

        public VertexShader VertexShader
        {
            get
            {
                return this.@vertexShader;
            }
        }

        public GeometryShader GeometryShader
        {
            get
            {
                return this.@geometryShader;
            }
        }

        public FragmentShader FragmentShader
        {
            get
            {
                return this.@fragmentShader;
            }
        }

        public ComputeShader ComputeShader
        {
            get
            {
                return this.@computeShader;
            }
        }

        public ObservableList<Macro> Macros
        {
            get
            {
                return this.@macros;
            }
        }

        public ObservableList<Variation> Variations
        {
            get
            {
                return this.@variations;
            }
        }

        public string Name
        {
            get
            {
                return this.@name;
            }

            set
            {
                if (this.@name == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@name, value, () => this.CalcCRC());
            }
        }

        public bool MaterialShader
        {
            get
            {
                return this.@materialShader;
            }

            set
            {
                if (this.@materialShader == value)
                {
                    return;
                }

                this.SetProperty(ref this.@materialShader, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override shaderType CreateSerializableData()
        {
            Ensure.Operation.True(
                this.vertexShader.IsEnabled
                || this.fragmentShader.IsEnabled
                || this.geometryShader.IsEnabled
                || this.computeShader.IsEnabled);
            var writeData = new shaderType();
            writeData.@vertex_shader = this.@vertexShader.CreateSerializableData() as vertex_shaderType;
            writeData.@geometry_shader = this.@geometryShader.CreateSerializableData() as geometry_shaderType;
            writeData.@fragment_shader = this.@fragmentShader.CreateSerializableData() as fragment_shaderType;
            writeData.@compute_shader = this.@computeShader.CreateSerializableData() as compute_shaderType;
            if (this.@Macros.Count > 0)
            {
                writeData.@macro_array = new macro_arrayType();
                writeData.@macro_array.Items = this.@macros.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@macro_array.Items.Length == 0)
                {
                    writeData.@macro_array = null;
                }
            }
            if (this.@Variations.Count > 0)
            {
                writeData.@variation_array = new variation_arrayType();
                writeData.@variation_array.Items = this.@variations.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@variation_array.Items.Length == 0)
                {
                    writeData.@variation_array = null;
                }
            }
            writeData.@name = this.Name;
            writeData.@material_shader = this.MaterialShader;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Shader IDeepCopyable<Shader>.DeepCopy()
        {
            return new Shader(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Shader source)
        {
            this.@vertexShader.DeepCopyFrom(source.@vertexShader);
            this.@geometryShader.DeepCopyFrom(source.@geometryShader);
            this.@fragmentShader.DeepCopyFrom(source.@fragmentShader);
            this.@computeShader.DeepCopyFrom(source.@computeShader);
            CopyElements(source.@macros, this.@macros);
            CopyElements(source.@variations, this.@variations);
            this.@name = source.@name;
            this.@materialShader = source.@materialShader;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.VertexShader.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.GeometryShader.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.FragmentShader.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ComputeShader.HashValue)));
            foreach (var elem in this.Macros)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.Variations)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Name)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.MaterialShader)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            this.@vertexShader.AutoCalc = this.AutoCalc;
            this.@geometryShader.AutoCalc = this.AutoCalc;
            this.@fragmentShader.AutoCalc = this.AutoCalc;
            this.@computeShader.AutoCalc = this.AutoCalc;
            foreach (var elem in this.@macros)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@variations)
            {
                elem.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            this.@vertexShader.Reset();
            this.@geometryShader.Reset();
            this.@fragmentShader.Reset();
            this.@computeShader.Reset();
            foreach (var elem in this.@macros)
            {
                elem.Reset();
            }
            foreach (var elem in this.@variations)
            {
                elem.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            this.@vertexShader.Refresh();
            this.@geometryShader.Refresh();
            this.@fragmentShader.Refresh();
            this.@computeShader.Refresh();
            foreach (var elem in this.@macros)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@variations)
            {
                elem.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
