﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class SamplerAssign : ObservableEntity<sampler_assignType>, IChildEntity, IDeepCopyable<SamplerAssign>, IDeepCopyFrom<SamplerAssign>
    {
        private string @id = string.Empty;
        private string @samplerName = string.Empty;
        private ShaderAssign parent = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SamplerAssign()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public SamplerAssign(SamplerAssign source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public SamplerAssign(sampler_assignType source)
        {
            this.@id = source.@id;
            this.@samplerName = source.@sampler_name;
        }

        public string Id
        {
            get
            {
                return this.@id;
            }

            set
            {
                if (this.@id == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@id, value, () => this.CalcCRC());
            }
        }

        public string SamplerName
        {
            get
            {
                return this.@samplerName;
            }

            set
            {
                if (this.@samplerName == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@samplerName, value, () => this.CalcCRC());
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is ShaderAssign : true);
                this.SetProperty(ref this.parent, value as ShaderAssign, () => this.CalcCRC());
            }
        }

        public ShaderAssign Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(ShaderAssign);
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override sampler_assignType CreateSerializableData()
        {
            var writeData = new sampler_assignType();
            writeData.@id = this.Id;
            writeData.@sampler_name = this.SamplerName;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        SamplerAssign IDeepCopyable<SamplerAssign>.DeepCopy()
        {
            return new SamplerAssign(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(SamplerAssign source)
        {
            this.@id = source.@id;
            this.@samplerName = source.@samplerName;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Id)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.SamplerName)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
