﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;
using System.Text.RegularExpressions;

namespace Nintendo.G3dTool.Entities
{
    public enum RenderInfoNumericChoiceType
    {
        Range,
        List,
    }

    public abstract class RenderInfoNumericChoice<TValue> : ObservableEntity<string>, IDeepCopyFrom<RenderInfoNumericChoice<TValue>>
        where TValue : struct
    {
        private RenderInfoNumericChoiceType type = RenderInfoNumericChoiceType.List;
        private TValue minValue;
        private TValue maxValue;

        public TValue MinValue
        {
            get
            {
                return this.minValue;
            }

            set
            {
                SetProperty<TValue>(ref this.minValue, value, () => CalcCRC());
            }
        }

        public TValue MaxValue
        {
            get
            {
                return this.maxValue;
            }

            set
            {
                SetProperty<TValue>(ref this.maxValue, value, () => CalcCRC());
            }
        }

        public RenderInfoNumericChoiceType Type
        {
            get
            {
                return this.type;
            }

            set
            {
                SetProperty<RenderInfoNumericChoiceType>(ref this.type, value, () => CalcCRC());
            }
        }

        public void DeepCopyFrom(RenderInfoNumericChoice<TValue> source)
        {
            this.minValue = source.minValue;
            this.maxValue = source.maxValue;
            this.type = source.type;
        }

        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Type)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.MinValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.MaxValue)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        protected void SetRange(TValue minValue, TValue maxValue)
        {
            this.minValue = minValue;
            this.maxValue = maxValue;
            this.type = RenderInfoNumericChoiceType.Range;
        }
    }

    public class RenderInfoIntChoice : RenderInfoNumericChoice<int>
    {
        private readonly ObservableList<ChoiceValueInt> values = new ObservableList<ChoiceValueInt>();

        public RenderInfoIntChoice()
        {
        }

        public RenderInfoIntChoice(string source)
        {
            this.SetFromShaderAnnotationFormat(source);
        }


        public ObservableList<ChoiceValueInt> Values
        {
            get
            {
                return this.values;
            }
        }

        public override string CreateSerializableData()
        {
            switch (this.Type)
            {
                case RenderInfoNumericChoiceType.List:
                    {
                        if (this.Values.Count == 0)
                        {
                            return string.Empty;
                        }

                        return string.Join(",", this.values.Select(x => x.ToString()));
                    }
                case RenderInfoNumericChoiceType.Range:
                    {
                        return $"[{this.MinValue}, {this.MaxValue}]";
                    }
                default:
                    throw new Exception("Unexpected default");
            }
        }

        public void DeepCopyFrom(RenderInfoIntChoice source)
        {
            base.DeepCopyFrom(source);
            CopyElements(source.values, this.values);
        }


        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(base.CreateCRCInternal()));
            foreach (var elem in this.Values)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(elem)));
            }
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        private void SetFromShaderAnnotationFormat(string source)
        {
            if (source.StartsWith("["))
            {
                Ensure.Argument.True(source.EndsWith("]"));

                string[] splited = source.Substring(1, source.Length - 2).Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
                Ensure.Argument.True(splited.Length == 2);

                int min, max;
                int.TryParse(splited[0], out min);
                int.TryParse(splited[1], out max);
                this.SetRange(min, max);
            }
            else
            {
                string[] splited = source.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
                this.Values.Clear();
                this.Values.Add(splited.Select(x => new ChoiceValueInt(x)));
                this.Type = RenderInfoNumericChoiceType.List;
            }
        }
    }

    public class RenderInfoFloatChoice : RenderInfoNumericChoice<float>, IDeepCopyFrom<RenderInfoFloatChoice>
    {
        private readonly ObservableList<ChoiceValueFloat> values = new ObservableList<ChoiceValueFloat>();

        public RenderInfoFloatChoice()
        {
        }

        public RenderInfoFloatChoice(string source)
        {
            this.SetFromShaderAnnotationFormat(source);
        }

        public ObservableList<ChoiceValueFloat> Values
        {
            get
            {
                return this.values;
            }
        }

        public void DeepCopyFrom(RenderInfoFloatChoice source)
        {
            base.DeepCopyFrom(source);
            CopyElements(source.values, this.values);
        }

        public override string CreateSerializableData()
        {
            switch (this.Type)
            {
                case RenderInfoNumericChoiceType.List:
                    {
                        if (this.Values.Count == 0)
                        {
                            return string.Empty;
                        }

                        return string.Join(",", this.values.Select(x => x.ToString()));
                    }
                case RenderInfoNumericChoiceType.Range:
                    {
                        return $"[{this.MinValue}, {this.MaxValue}]";
                    }
                default:
                    throw new Exception("Unexpected default");
            }
        }

        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(base.CreateCRCInternal()));
            foreach (var elem in this.Values)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(elem)));
            }
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        private void SetFromShaderAnnotationFormat(string source)
        {
            if (source.StartsWith("["))
            {
                Ensure.Argument.True(source.EndsWith("]"));

                string[] splited = source.Substring(1, source.Length - 2).Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
                Ensure.Argument.True(splited.Length == 2);

                float min, max;
                float.TryParse(splited[0], out min);
                float.TryParse(splited[1], out max);
                this.SetRange(min, max);
            }
            else
            {
                string[] splited = source.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
                this.Values.Clear();
                this.Values.Add(splited.Select(x => new ChoiceValueFloat(x)));
                this.Type = RenderInfoNumericChoiceType.List;
            }
        }
    }
}
