﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public abstract class RenderInfo : ObservableEntity<render_infoType>, IDeepCopyable<RenderInfo>, IDeepCopyFrom<RenderInfo>
    {
        private string @name = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public RenderInfo()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public RenderInfo(RenderInfo source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public RenderInfo(render_infoType source)
        {
            this.@name = source.@name;
        }

        public string Name
        {
            get
            {
                return this.@name;
            }

            set
            {
                if (this.@name == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@name, value, () => this.CalcCRC());
            }
        }

        public abstract int Count{ get; }

        public abstract render_info_typeType Type{ get; }

        private protected abstract void SetToDefaultInternal(RenderInfoSlot definition);

        private protected void SetToDefault(RenderInfoSlot definition)
        {
            Ensure.Argument.AreEqual(this.Name, definition.Name);
            this.SetToDefaultInternal(definition);
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override render_infoType CreateSerializableData()
        {
            var writeData = new render_infoType();
            writeData.@name = this.Name;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        RenderInfo IDeepCopyable<RenderInfo>.DeepCopy()
        {
            return this.DeepCopy();
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        private protected abstract RenderInfo DeepCopy();

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(RenderInfo source)
        {
            this.@name = source.@name;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Name)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
