﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class ProcessLog : ObservableEntity<process_logType>, IDeepCopyable<ProcessLog>, IDeepCopyFrom<ProcessLog>
    {
        private string @process = "undefined";
        private string @argument = string.Empty;
        private string @value = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ProcessLog()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ProcessLog(ProcessLog source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ProcessLog(process_logType source)
        {
            this.@process = source.@process;
            this.@argument = source.@argument;
            this.@value = source.@Value;
        }

        public string Process
        {
            get
            {
                return this.@process;
            }

            set
            {
                if (this.@process == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@process, value, () => this.CalcCRC());
            }
        }

        public string Argument
        {
            get
            {
                return this.@argument;
            }

            set
            {
                if (this.@argument == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@argument, value, () => this.CalcCRC());
            }
        }

        public string Value
        {
            get
            {
                return this.@value;
            }

            set
            {
                if (this.@value == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@value, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override process_logType CreateSerializableData()
        {
            var writeData = new process_logType();
            writeData.@process = this.Process;
            writeData.@argument = this.Argument;
            writeData.@Value = this.Value;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        ProcessLog IDeepCopyable<ProcessLog>.DeepCopy()
        {
            return new ProcessLog(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(ProcessLog source)
        {
            this.@process = source.@process;
            this.@argument = source.@argument;
            this.@value = source.@value;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Process)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Argument)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Value)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
