﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class OriginalPerMaterialAnim : ObservableEntity<original_material_animType>, IDeepCopyable<OriginalPerMaterialAnim>, IDeepCopyFrom<OriginalPerMaterialAnim>
    {
        private readonly ObservableList<OriginalColorAnim> @originalColorAnims = new ObservableList<OriginalColorAnim>();
        private readonly ObservableList<OriginalTexsrtAnim> @originalTexsrtAnims = new ObservableList<OriginalTexsrtAnim>();
        private string @matName = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public OriginalPerMaterialAnim()
        {
            this.@originalColorAnims.CollectionChanged += OnCollectionChanged<OriginalColorAnim>;
            this.@originalTexsrtAnims.CollectionChanged += OnCollectionChanged<OriginalTexsrtAnim>;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public OriginalPerMaterialAnim(OriginalPerMaterialAnim source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public OriginalPerMaterialAnim(original_material_animType source)
        {
            this.@originalColorAnims.CollectionChanged += OnCollectionChanged<OriginalColorAnim>;
            this.@originalTexsrtAnims.CollectionChanged += OnCollectionChanged<OriginalTexsrtAnim>;
            if (source.@original_color_anim_array != null)
            {
                this.@originalColorAnims.Clear();
                foreach (var elem in source.@original_color_anim_array.Items)
                {
                    this.@originalColorAnims.Add(new OriginalColorAnim(elem));
                }
            }
            if (source.@original_texsrt_anim_array != null)
            {
                this.@originalTexsrtAnims.Clear();
                foreach (var elem in source.@original_texsrt_anim_array.Items)
                {
                    this.@originalTexsrtAnims.Add(new OriginalTexsrtAnim(elem));
                }
            }
            this.@matName = source.@mat_name;
        }

        public ObservableList<OriginalColorAnim> OriginalColorAnims
        {
            get
            {
                return this.@originalColorAnims;
            }
        }

        public ObservableList<OriginalTexsrtAnim> OriginalTexsrtAnims
        {
            get
            {
                return this.@originalTexsrtAnims;
            }
        }

        public string MatName
        {
            get
            {
                return this.@matName;
            }

            set
            {
                if (this.@matName == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@matName, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override original_material_animType CreateSerializableData()
        {
            var writeData = new original_material_animType();
            if (this.@OriginalColorAnims.Count > 0)
            {
                writeData.@original_color_anim_array = new original_color_anim_arrayType();
                writeData.@original_color_anim_array.Items = this.@originalColorAnims.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@original_color_anim_array.Items.Length == 0)
                {
                    writeData.@original_color_anim_array = null;
                }
            }
            if (this.@OriginalTexsrtAnims.Count > 0)
            {
                writeData.@original_texsrt_anim_array = new original_texsrt_anim_arrayType();
                writeData.@original_texsrt_anim_array.Items = this.@originalTexsrtAnims.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@original_texsrt_anim_array.Items.Length == 0)
                {
                    writeData.@original_texsrt_anim_array = null;
                }
            }
            writeData.@mat_name = this.MatName;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        OriginalPerMaterialAnim IDeepCopyable<OriginalPerMaterialAnim>.DeepCopy()
        {
            return new OriginalPerMaterialAnim(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(OriginalPerMaterialAnim source)
        {
            CopyElements(source.@originalColorAnims, this.@originalColorAnims);
            CopyElements(source.@originalTexsrtAnims, this.@originalTexsrtAnims);
            this.@matName = source.@matName;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.OriginalColorAnims)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.OriginalTexsrtAnims)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.MatName)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@originalColorAnims)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@originalTexsrtAnims)
            {
                elem.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@originalColorAnims)
            {
                elem.Reset();
            }
            foreach (var elem in this.@originalTexsrtAnims)
            {
                elem.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@originalColorAnims)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@originalTexsrtAnims)
            {
                elem.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
