﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class OriginalColorAnim : ObservableEntity<original_color_animType>, IDeepCopyable<OriginalColorAnim>, IDeepCopyFrom<OriginalColorAnim>
    {
        private readonly ObservableList<OriginalColorAnimTarget> @originalColorAnimTargets = new ObservableList<OriginalColorAnimTarget>();
        private string @hint = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public OriginalColorAnim()
        {
            this.@originalColorAnimTargets.CollectionChanged += OnCollectionChanged<OriginalColorAnimTarget>;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public OriginalColorAnim(OriginalColorAnim source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public OriginalColorAnim(original_color_animType source)
        {
            this.@originalColorAnimTargets.CollectionChanged += OnCollectionChanged<OriginalColorAnimTarget>;
            if (source.@original_color_anim_target != null)
            {
                this.@originalColorAnimTargets.Clear();
                foreach (var elem in source.@original_color_anim_target)
                {
                    this.@originalColorAnimTargets.Add(new OriginalColorAnimTarget(elem));
                }
            }
            this.@hint = source.@hint;
        }

        public ObservableList<OriginalColorAnimTarget> OriginalColorAnimTargets
        {
            get
            {
                return this.@originalColorAnimTargets;
            }
        }

        public string Hint
        {
            get
            {
                return this.@hint;
            }

            set
            {
                if (this.@hint == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@hint, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override original_color_animType CreateSerializableData()
        {
            var writeData = new original_color_animType();
            if (this.@OriginalColorAnimTargets.Count > 0)
            {
                writeData.@original_color_anim_target = this.@originalColorAnimTargets.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@original_color_anim_target.Length == 0)
                {
                    writeData.@original_color_anim_target = null;
                }
            }
            writeData.@hint = this.Hint;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        OriginalColorAnim IDeepCopyable<OriginalColorAnim>.DeepCopy()
        {
            return new OriginalColorAnim(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(OriginalColorAnim source)
        {
            CopyElements(source.@originalColorAnimTargets, this.@originalColorAnimTargets);
            this.@hint = source.@hint;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.OriginalColorAnimTargets)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Hint)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@originalColorAnimTargets)
            {
                elem.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@originalColorAnimTargets)
            {
                elem.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@originalColorAnimTargets)
            {
                elem.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
