﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace Nintendo.G3dTool.Entities
{
    using Nintendo.ToolFoundation.Collections;
    using System;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.ComponentModel;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using Internal;

    public abstract class ObservableEntity<TEntity> : Entity<TEntity> where TEntity : class
    {
        private protected void OnCollectionChanged<TItem>(object sender, NotifyCollectionChangedEventArgs e) where TItem : INotifyPropertyChanged
        {
            if (e.NewItems != null)
            {
                foreach (var item in e.NewItems)
                {
                    TItem observableItem = (TItem)item;
                    observableItem.PropertyChanged += this.OnCollectionPropertyChanged;
                    var child = observableItem as IChildEntity;

                    // 構造的な親の関連付け。同じ型が構造的に親になることはないので同じ型は省く
                    if (child != null && this.GetType().IsAssignableFrom(child.ParentType))
                    {
                        child.Parent = this;
                    }
                }
            }

            if (e.OldItems != null)
            {
                foreach (var item in e.OldItems)
                {
                    TItem observableItem = (TItem)item;
                    observableItem.PropertyChanged -= this.OnCollectionPropertyChanged;
                    var child = observableItem as IChildEntity;

                    // 構造的な親の関連付け。同じ型が構造的に親になることはないので同じ型は省く
                    if (child != null && this.GetType().IsAssignableFrom(child.ParentType))
                    {
                        child.Parent = null;
                    }
                }
            }

            this.CalcCRC();
        }

        private protected void OnDictionaryChanged<TKey, TItem>(object sender, NotifyCollectionChangedEventArgs e) where TItem : INotifyPropertyChanged
        {
            if (e.NewItems != null)
            {
                foreach (var item in e.NewItems)
                {
                    var keyValuePair = (KeyValuePair<TKey, TItem>)item;
                    TItem observableItem = (TItem)keyValuePair.Value;
                    observableItem.PropertyChanged += this.OnCollectionPropertyChanged;
                    var child = observableItem as IChildEntity;
                    if (child != null && child.GetType() != this.GetType())
                    {
                        child.Parent = this;
                    }
                }
            }

            if (e.OldItems != null)
            {
                foreach (var item in e.OldItems)
                {
                    var keyValuePair = (KeyValuePair<TKey, TItem>)item;
                    TItem observableItem = (TItem)keyValuePair.Value;
                    observableItem.PropertyChanged -= this.OnCollectionPropertyChanged;
                    var child = observableItem as IChildEntity;
                    if (child != null && child.GetType() != this.GetType())
                    {
                        child.Parent = null;
                    }
                }
            }

            this.CalcCRC();
        }

        private protected void OnValueCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            this.CalcCRC();
        }

        private protected void CopyElements<T>(ObservableList<T> source, ObservableList<T> destination)
            where T : IDeepCopyable<T>
        {
            destination.Clear();
            foreach (var value in source)
            {
                destination.Add(value.DeepCopy());
            }
        }

        private protected void CopyValues<T>(IEnumerable<T> source, ObservableList<T> destination)
        {
            destination.Clear();
            foreach (var value in source)
            {
                destination.Add(value);
            }
        }

        private protected void SetValues(IEnumerable<float> source, float[] destination)
        {
            for (int elemIndex = 0; elemIndex < destination.Length; ++elemIndex)
            {
                destination[elemIndex] = source.ElementAtOrDefault(elemIndex);
            }
        }

        private protected void SetValues(IEnumerable<float> source, bool[] destination)
        {
            for (int elemIndex = 0; elemIndex < destination.Length; ++elemIndex)
            {
                destination[elemIndex] = ((int)source.ElementAtOrDefault(elemIndex)) != 0;
            }
        }

        private protected void SetValues(IEnumerable<float> source, int[] destination)
        {
            for (int elemIndex = 0; elemIndex < destination.Length; ++elemIndex)
            {
                destination[elemIndex] = (int)source.ElementAtOrDefault(elemIndex);
            }
        }

        private protected void SetValues(IEnumerable<float> source, uint[] destination)
        {
            for (int elemIndex = 0; elemIndex < destination.Length; ++elemIndex)
            {
                destination[elemIndex] = (uint)source.ElementAtOrDefault(elemIndex);
            }
        }

        private protected void CopyElements<TKey, TValue>(ObservableDictionary<TKey, TValue> source, ObservableDictionary<TKey, TValue> destination)
            where TValue : IDeepCopyable<TValue>
        {
            destination.Clear();
            foreach (var value in source)
            {
                destination.Add(value.Key, value.Value.DeepCopy());
            }
        }

        private void OnCollectionPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
