﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Model : RootEntity<modelType>, IDeepCopyable<Model>, IDeepCopyFrom<Model>, IToolData
    {
        private readonly ObservableList<ProcessLog> @processLogs = new ObservableList<ProcessLog>();
        private readonly ModelInfo @modelInfo = new ModelInfo();
        private readonly ObservableList<Material> @materials = new ObservableList<Material>();
        private readonly Skeleton @skeleton = new Skeleton();
        private readonly ObservableList<Vertex> @vertexes = new ObservableList<Vertex>();
        private readonly ObservableList<Shape> @shapes = new ObservableList<Shape>();
        private readonly ObservableList<OriginalMaterial> @originalMaterials = new ObservableList<OriginalMaterial>();
        private readonly ObservableList<UserData> @userDatas = new ObservableList<UserData>();
        private readonly ObservableList<Stream> @streams = new ObservableList<Stream>();
        private readonly Comment @comment = new Comment();
        private readonly ToolData @toolData = new ToolData();
        private readonly UserToolData @userToolData = new UserToolData();
        private model_versionType @version;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        internal Model()
        {
            this.@processLogs.CollectionChanged += OnCollectionChanged<ProcessLog>;
            this.@modelInfo.PropertyChanged += this.OnPropertyChanged;
            this.@materials.CollectionChanged += OnCollectionChanged<Material>;
            this.@skeleton.PropertyChanged += this.OnPropertyChanged;
            this.@vertexes.CollectionChanged += OnCollectionChanged<Vertex>;
            this.@shapes.CollectionChanged += OnCollectionChanged<Shape>;
            this.@originalMaterials.CollectionChanged += OnCollectionChanged<OriginalMaterial>;
            this.@userDatas.CollectionChanged += OnCollectionChanged<UserData>;
            this.@streams.CollectionChanged += OnCollectionChanged<Stream>;
            this.@comment.PropertyChanged += this.OnPropertyChanged;
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
            (this.@modelInfo as IChildEntity).Parent = this;
            (this.@skeleton as IChildEntity).Parent = this;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Model(Model source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Model(modelType source)
        {
            this.@processLogs.CollectionChanged += OnCollectionChanged<ProcessLog>;
            this.@materials.CollectionChanged += OnCollectionChanged<Material>;
            this.@vertexes.CollectionChanged += OnCollectionChanged<Vertex>;
            this.@shapes.CollectionChanged += OnCollectionChanged<Shape>;
            this.@originalMaterials.CollectionChanged += OnCollectionChanged<OriginalMaterial>;
            this.@userDatas.CollectionChanged += OnCollectionChanged<UserData>;
            this.@streams.CollectionChanged += OnCollectionChanged<Stream>;
            if (source.@process_log_array != null)
            {
                this.@processLogs.Clear();
                foreach (var elem in source.@process_log_array.Items)
                {
                    this.@processLogs.Add(new ProcessLog(elem));
                }
            }
            if (source.@model_info != null)
            {
                this.@modelInfo = new ModelInfo(source.@model_info);
            }
            else
            {
                this.@modelInfo = new ModelInfo();
            }
            (this.@modelInfo as IChildEntity).Parent = this;
            if (source.@material_array != null)
            {
                this.@materials.Clear();
                foreach (var elem in source.@material_array.Items)
                {
                    this.@materials.Add(new Material(elem));
                }
            }
            if (source.@skeleton != null)
            {
                this.@skeleton = new Skeleton(source.@skeleton);
            }
            else
            {
                this.@skeleton = new Skeleton();
            }
            (this.@skeleton as IChildEntity).Parent = this;
            if (source.@vertex_array != null)
            {
                this.@vertexes.Clear();
                foreach (var elem in source.@vertex_array.Items)
                {
                    this.@vertexes.Add(new Vertex(elem));
                }
            }
            if (source.@shape_array != null)
            {
                this.@shapes.Clear();
                foreach (var elem in source.@shape_array.Items)
                {
                    this.@shapes.Add(new Shape(elem));
                }
            }
            if (source.@original_material_array != null)
            {
                this.@originalMaterials.Clear();
                foreach (var elem in source.@original_material_array.Items)
                {
                    this.@originalMaterials.Add(new OriginalMaterial(elem));
                }
            }
            if (source.@user_data_array != null)
            {
                this.@userDatas.Clear();
                foreach (var elem in source.@user_data_array.Items)
                {
                    this.@userDatas.Add(DataModelConverter.Convert(elem));
                }
            }
            if (source.@stream_array != null)
            {
                this.@streams.Clear();
                foreach (var elem in source.@stream_array.Items)
                {
                    this.@streams.Add(DataModelConverter.Convert(elem));
                }
            }
            if (source.@comment != null)
            {
                this.@comment = new Comment(source.@comment);
            }
            else
            {
                this.@comment = new Comment();
            }
            if (source.@tool_data != null)
            {
                this.@toolData = new ToolData(source.@tool_data);
            }
            else
            {
                this.@toolData = new ToolData();
            }
            if (source.@user_tool_data != null)
            {
                this.@userToolData = new UserToolData(source.@user_tool_data);
            }
            else
            {
                this.@userToolData = new UserToolData();
            }
            this.@version = source.@version;
            this.@modelInfo.PropertyChanged += this.OnPropertyChanged;
            this.@skeleton.PropertyChanged += this.OnPropertyChanged;
            this.@comment.PropertyChanged += this.OnPropertyChanged;
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
            this.ResolveInternalReferences();
        }

        public ObservableList<ProcessLog> ProcessLogs
        {
            get
            {
                return this.@processLogs;
            }
        }

        public ModelInfo ModelInfo
        {
            get
            {
                return this.@modelInfo;
            }
        }

        public ObservableList<Material> Materials
        {
            get
            {
                return this.@materials;
            }
        }

        public Skeleton Skeleton
        {
            get
            {
                return this.@skeleton;
            }
        }

        public ObservableList<Vertex> Vertexes
        {
            get
            {
                return this.@vertexes;
            }
        }

        public ObservableList<Shape> Shapes
        {
            get
            {
                return this.@shapes;
            }
        }

        public ObservableList<OriginalMaterial> OriginalMaterials
        {
            get
            {
                return this.@originalMaterials;
            }
        }

        public ObservableList<UserData> UserDatas
        {
            get
            {
                return this.@userDatas;
            }
        }

        public override ObservableList<Stream> Streams
        {
            get
            {
                return this.@streams;
            }
        }

        public Comment Comment
        {
            get
            {
                return this.@comment;
            }
        }

        public ToolData ToolData
        {
            get
            {
                return this.@toolData;
            }
        }

        public UserToolData UserToolData
        {
            get
            {
                return this.@userToolData;
            }
        }

        public model_versionType Version
        {
            get
            {
                return this.@version;
            }

            set
            {
                if (this.@version == value)
                {
                    return;
                }

                this.SetProperty(ref this.@version, value, () => this.CalcCRC());
            }
        }

        public override void ResolveInternalReferences()
        {
            // インデックスで紐づけをしているデータに実データの割り当て
            if (this.Streams.Count > 0)
            {
                foreach (Shape shape in this.Shapes)
                {
                    // メッシュからストリームへの参照設定
                    foreach (Mesh mesh in shape.Meshes)
                    {
                        Ensure.Operation.Range((mesh as IStreamIndex).StreamIndex, 0, this.Streams.Count, $"{0} <= {(mesh as IStreamIndex).StreamIndex} < {this.Streams.Count}");
                        mesh.IndexStream = this.Streams[(mesh as IStreamIndex).StreamIndex] as StreamInt;
                    }
                }

                // 頂点属性からストリームへの参照設定
                foreach (Vertex vertex in this.Vertexes)
                {
                    foreach (VtxAttrib attr in vertex.VtxAttribs)
                    {
                        Ensure.Operation.Range((attr as IStreamIndex).StreamIndex, 0, this.Streams.Count, $"{0} <= {(attr as IStreamIndex).StreamIndex} < {this.Streams.Count}");
                        attr.Stream = this.Streams[(attr as IStreamIndex).StreamIndex];
                    }
                }
            }
            else
            {
                // 高速にファイルを読むためにストリームを読み飛ばす場合もあるので、ストリームがない場合は何もしない
            }

            foreach (Shape shape in this.Shapes)
            {
                // シェイプから頂点属性への参照設定
                Ensure.Operation.Range(shape.ShapeInfo.VertexIndex, 0, this.Vertexes.Count, $"{0} <= {shape.ShapeInfo.VertexIndex} < {this.Vertexes.Count}");
                shape.ShapeInfo.Vertex = this.Vertexes[shape.ShapeInfo.VertexIndex];
                shape.ShapeInfo.Bone = this.Skeleton.EnumerateBones().First(x => x.Name == shape.ShapeInfo.BoneName);
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override modelType CreateSerializableData()
        {
            var writeData = new modelType();
            if (this.@ProcessLogs.Count > 0)
            {
                writeData.@process_log_array = new process_log_arrayType();
                writeData.@process_log_array.Items = this.@processLogs.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@process_log_array.Items.Length == 0)
                {
                    writeData.@process_log_array = null;
                }
            }
            writeData.@model_info = this.@modelInfo.CreateSerializableData() as model_infoType;
            if (this.@Materials.Count > 0)
            {
                writeData.@material_array = new material_arrayType();
                writeData.@material_array.Items = this.@materials.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@material_array.Items.Length == 0)
                {
                    writeData.@material_array = null;
                }
            }
            writeData.@skeleton = this.@skeleton.CreateSerializableData() as skeletonType;
            if (this.@Vertexes.Count > 0)
            {
                writeData.@vertex_array = new vertex_arrayType();
                writeData.@vertex_array.Items = this.@vertexes.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@vertex_array.Items.Length == 0)
                {
                    writeData.@vertex_array = null;
                }
            }
            if (this.@Shapes.Count > 0)
            {
                writeData.@shape_array = new shape_arrayType();
                writeData.@shape_array.Items = this.@shapes.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@shape_array.Items.Length == 0)
                {
                    writeData.@shape_array = null;
                }
            }
            if (this.@OriginalMaterials.Count > 0)
            {
                writeData.@original_material_array = new original_material_arrayType();
                writeData.@original_material_array.Items = this.@originalMaterials.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@original_material_array.Items.Length == 0)
                {
                    writeData.@original_material_array = null;
                }
            }
            if (this.@UserDatas.Count > 0)
            {
                writeData.@user_data_array = new user_data_arrayType();
                writeData.@user_data_array.Items = this.@userDatas.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@user_data_array.Items.Length == 0)
                {
                    writeData.@user_data_array = null;
                }
            }
            if (this.IsStreamSerializationEnabled)
            {
                if (this.@Streams.Count > 0)
                {
                    writeData.@stream_array = new stream_arrayType();
                    writeData.@stream_array.Items = this.@streams.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                    if (writeData.@stream_array.Items.Length == 0)
                    {
                        writeData.@stream_array = null;
                    }
                }
            }
            writeData.@comment = this.@comment.CreateSerializableData() as commentType;
            writeData.@tool_data = this.@toolData.CreateSerializableData() as tool_dataType;
            writeData.@user_tool_data = this.@userToolData.CreateSerializableData() as user_tool_dataType;
            writeData.@version = this.Version;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Model IDeepCopyable<Model>.DeepCopy()
        {
            return new Model(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Model source)
        {
            CopyElements(source.@processLogs, this.@processLogs);
            this.@modelInfo.DeepCopyFrom(source.@modelInfo);
            CopyElements(source.@materials, this.@materials);
            this.@skeleton.DeepCopyFrom(source.@skeleton);
            CopyElements(source.@vertexes, this.@vertexes);
            CopyElements(source.@shapes, this.@shapes);
            CopyElements(source.@originalMaterials, this.@originalMaterials);
            CopyElements(source.@userDatas, this.@userDatas);
            CopyElements(source.@streams, this.@streams);
            this.@comment.DeepCopyFrom(source.@comment);
            this.@toolData.DeepCopyFrom(source.@toolData);
            this.@userToolData.DeepCopyFrom(source.@userToolData);
            this.@version = source.@version;
            this.ResolveInternalReferences();
        }

        private protected override IRootEntity DeepCopy()
        {
            return new Model(this);
        }

        private protected override void DeepCopyFrom(IRootEntity source)
        {
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.True(source is Model);
            this.DeepCopyFrom(source as Model);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.ProcessLogs)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ModelInfo.HashValue)));
            foreach (var elem in this.Materials)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Skeleton.HashValue)));
            foreach (var elem in this.Vertexes)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.Shapes)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.OriginalMaterials)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.UserDatas)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.Streams)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Comment.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ToolData.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.UserToolData.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Version)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@processLogs)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            this.@modelInfo.AutoCalc = this.AutoCalc;
            foreach (var elem in this.@materials)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            this.@skeleton.AutoCalc = this.AutoCalc;
            foreach (var elem in this.@vertexes)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@shapes)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@originalMaterials)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@userDatas)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@streams)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            this.@comment.AutoCalc = this.AutoCalc;
            this.@toolData.AutoCalc = this.AutoCalc;
            this.@userToolData.AutoCalc = this.AutoCalc;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@processLogs)
            {
                elem.Reset();
            }
            this.@modelInfo.Reset();
            foreach (var elem in this.@materials)
            {
                elem.Reset();
            }
            this.@skeleton.Reset();
            foreach (var elem in this.@vertexes)
            {
                elem.Reset();
            }
            foreach (var elem in this.@shapes)
            {
                elem.Reset();
            }
            foreach (var elem in this.@originalMaterials)
            {
                elem.Reset();
            }
            foreach (var elem in this.@userDatas)
            {
                elem.Reset();
            }
            foreach (var elem in this.@streams)
            {
                elem.Reset();
            }
            this.@comment.Reset();
            this.@toolData.Reset();
            this.@userToolData.Reset();
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@processLogs)
            {
                elem.Refresh();
            }
            this.@modelInfo.Refresh();
            foreach (var elem in this.@materials)
            {
                elem.Refresh();
            }
            this.@skeleton.Refresh();
            foreach (var elem in this.@vertexes)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@shapes)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@originalMaterials)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@userDatas)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@streams)
            {
                elem.Refresh();
            }
            this.@comment.Refresh();
            this.@toolData.Refresh();
            this.@userToolData.Refresh();
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
