﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Material : RootEntity<materialType>, IDeepCopyable<Material>, IDeepCopyFrom<Material>, IToolData
    {
        private readonly ObservableList<ProcessLog> @processLogs = new ObservableList<ProcessLog>();
        private readonly MaterialInfo @materialInfo = new MaterialInfo();
        private readonly ObservableList<Sampler> @samplers = new ObservableList<Sampler>();
        private readonly ShaderAssign @shaderAssign = new ShaderAssign();
        private readonly ObservableList<UserData> @userDatas = new ObservableList<UserData>();
        private readonly ObservableList<Stream> @streams = new ObservableList<Stream>();
        private readonly Comment @comment = new Comment();
        private readonly ToolData @toolData = new ToolData();
        private readonly UserToolData @userToolData = new UserToolData();
        private string @name = string.Empty;
        private material_versionType @version;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Material()
        {
            this.@processLogs.CollectionChanged += OnCollectionChanged<ProcessLog>;
            this.@materialInfo.PropertyChanged += this.OnPropertyChanged;
            this.@samplers.CollectionChanged += OnCollectionChanged<Sampler>;
            this.@shaderAssign.PropertyChanged += this.OnPropertyChanged;
            this.@userDatas.CollectionChanged += OnCollectionChanged<UserData>;
            this.@streams.CollectionChanged += OnCollectionChanged<Stream>;
            this.@comment.PropertyChanged += this.OnPropertyChanged;
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Material(Material source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Material(materialType source)
        {
            this.@processLogs.CollectionChanged += OnCollectionChanged<ProcessLog>;
            this.@samplers.CollectionChanged += OnCollectionChanged<Sampler>;
            this.@userDatas.CollectionChanged += OnCollectionChanged<UserData>;
            this.@streams.CollectionChanged += OnCollectionChanged<Stream>;
            if (source.@process_log_array != null)
            {
                this.@processLogs.Clear();
                foreach (var elem in source.@process_log_array.Items)
                {
                    this.@processLogs.Add(new ProcessLog(elem));
                }
            }
            if (source.@material_info != null)
            {
                this.@materialInfo = new MaterialInfo(source.@material_info);
            }
            else
            {
                this.@materialInfo = new MaterialInfo();
            }
            if (source.@sampler_array != null)
            {
                this.@samplers.Clear();
                foreach (var elem in source.@sampler_array.Items)
                {
                    this.@samplers.Add(new Sampler(elem));
                }
            }
            if (source.@shader_assign != null)
            {
                this.@shaderAssign = new ShaderAssign(source.@shader_assign);
            }
            else
            {
                this.@shaderAssign = new ShaderAssign();
            }
            if (source.@user_data_array != null)
            {
                this.@userDatas.Clear();
                foreach (var elem in source.@user_data_array.Items)
                {
                    this.@userDatas.Add(DataModelConverter.Convert(elem));
                }
            }
            if (source.@stream_array != null)
            {
                this.@streams.Clear();
                foreach (var elem in source.@stream_array.Items)
                {
                    this.@streams.Add(DataModelConverter.Convert(elem));
                }
            }
            if (source.@comment != null)
            {
                this.@comment = new Comment(source.@comment);
            }
            else
            {
                this.@comment = new Comment();
            }
            if (source.@tool_data != null)
            {
                this.@toolData = new ToolData(source.@tool_data);
            }
            else
            {
                this.@toolData = new ToolData();
            }
            if (source.@user_tool_data != null)
            {
                this.@userToolData = new UserToolData(source.@user_tool_data);
            }
            else
            {
                this.@userToolData = new UserToolData();
            }
            this.@name = source.@name;
            this.@version = source.@version;
            this.@materialInfo.PropertyChanged += this.OnPropertyChanged;
            this.@shaderAssign.PropertyChanged += this.OnPropertyChanged;
            this.@comment.PropertyChanged += this.OnPropertyChanged;
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
            this.ResolveInternalReferences();
        }

        public ObservableList<ProcessLog> ProcessLogs
        {
            get
            {
                return this.@processLogs;
            }
        }

        public MaterialInfo MaterialInfo
        {
            get
            {
                return this.@materialInfo;
            }
        }

        public ObservableList<Sampler> Samplers
        {
            get
            {
                return this.@samplers;
            }
        }

        public ShaderAssign ShaderAssign
        {
            get
            {
                return this.@shaderAssign;
            }
        }

        public ObservableList<UserData> UserDatas
        {
            get
            {
                return this.@userDatas;
            }
        }

        public override ObservableList<Stream> Streams
        {
            get
            {
                return this.@streams;
            }
        }

        public Comment Comment
        {
            get
            {
                return this.@comment;
            }
        }

        public ToolData ToolData
        {
            get
            {
                return this.@toolData;
            }
        }

        public UserToolData UserToolData
        {
            get
            {
                return this.@userToolData;
            }
        }

        public string Name
        {
            get
            {
                return this.@name;
            }

            set
            {
                if (this.@name == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@name, value, () => this.CalcCRC());
            }
        }

        public material_versionType Version
        {
            get
            {
                return this.@version;
            }

            set
            {
                if (this.@version == value)
                {
                    return;
                }

                this.SetProperty(ref this.@version, value, () => this.CalcCRC());
            }
        }

        public bool IsShaderAssigned
        {
            get
            {
                return !string.IsNullOrEmpty(ShaderAssign.ShaderArchive);
            }
        }

        public override void ResolveInternalReferences()
        {
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override materialType CreateSerializableData()
        {
            var writeData = new materialType();
            if (this.@ProcessLogs.Count > 0)
            {
                writeData.@process_log_array = new process_log_arrayType();
                writeData.@process_log_array.Items = this.@processLogs.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@process_log_array.Items.Length == 0)
                {
                    writeData.@process_log_array = null;
                }
            }
            writeData.@material_info = this.@materialInfo.CreateSerializableData() as material_infoType;
            if (this.@Samplers.Count > 0)
            {
                writeData.@sampler_array = new sampler_arrayType();
                writeData.@sampler_array.Items = this.@samplers.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@sampler_array.Items.Length == 0)
                {
                    writeData.@sampler_array = null;
                }
            }
            writeData.@shader_assign = this.@shaderAssign.CreateSerializableData() as shader_assignType;
            if (this.@UserDatas.Count > 0)
            {
                writeData.@user_data_array = new user_data_arrayType();
                writeData.@user_data_array.Items = this.@userDatas.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@user_data_array.Items.Length == 0)
                {
                    writeData.@user_data_array = null;
                }
            }
            if (this.IsStreamSerializationEnabled)
            {
                if (this.@Streams.Count > 0)
                {
                    writeData.@stream_array = new stream_arrayType();
                    writeData.@stream_array.Items = this.@streams.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                    if (writeData.@stream_array.Items.Length == 0)
                    {
                        writeData.@stream_array = null;
                    }
                }
            }
            writeData.@comment = this.@comment.CreateSerializableData() as commentType;
            writeData.@tool_data = this.@toolData.CreateSerializableData() as tool_dataType;
            writeData.@user_tool_data = this.@userToolData.CreateSerializableData() as user_tool_dataType;
            writeData.@name = this.Name;
            writeData.@version = this.Version;
            writeData.versionSpecified = true;

            // render_state は廃止予定なのでダミーだけ作成
            writeData.render_state = new render_stateType()
            {
                alpha_test = new alpha_testType(),
                color_blend = new color_blendType()
                {
                    const_color = new float[4],
                },
                depth_test = new depth_testType(),
                display_face = new render_state_display_faceType(),
                logical_blend = new logical_blendType(),
            };
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Material IDeepCopyable<Material>.DeepCopy()
        {
            return new Material(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Material source)
        {
            CopyElements(source.@processLogs, this.@processLogs);
            this.@materialInfo.DeepCopyFrom(source.@materialInfo);
            CopyElements(source.@samplers, this.@samplers);
            this.@shaderAssign.DeepCopyFrom(source.@shaderAssign);
            CopyElements(source.@userDatas, this.@userDatas);
            CopyElements(source.@streams, this.@streams);
            this.@comment.DeepCopyFrom(source.@comment);
            this.@toolData.DeepCopyFrom(source.@toolData);
            this.@userToolData.DeepCopyFrom(source.@userToolData);
            this.@name = source.@name;
            this.@version = source.@version;
            this.ResolveInternalReferences();
        }

        private protected override IRootEntity DeepCopy()
        {
            return new Material(this);
        }

        private protected override void DeepCopyFrom(IRootEntity source)
        {
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.True(source is Material);
            this.DeepCopyFrom(source as Material);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.ProcessLogs)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.MaterialInfo.HashValue)));
            foreach (var elem in this.Samplers)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ShaderAssign.HashValue)));
            foreach (var elem in this.UserDatas)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.Streams)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Comment.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ToolData.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.UserToolData.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Name)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Version)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@processLogs)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            this.@materialInfo.AutoCalc = this.AutoCalc;
            foreach (var elem in this.@samplers)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            this.@shaderAssign.AutoCalc = this.AutoCalc;
            foreach (var elem in this.@userDatas)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@streams)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            this.@comment.AutoCalc = this.AutoCalc;
            this.@toolData.AutoCalc = this.AutoCalc;
            this.@userToolData.AutoCalc = this.AutoCalc;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@processLogs)
            {
                elem.Reset();
            }
            this.@materialInfo.Reset();
            foreach (var elem in this.@samplers)
            {
                elem.Reset();
            }
            this.@shaderAssign.Reset();
            foreach (var elem in this.@userDatas)
            {
                elem.Reset();
            }
            foreach (var elem in this.@streams)
            {
                elem.Reset();
            }
            this.@comment.Reset();
            this.@toolData.Reset();
            this.@userToolData.Reset();
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@processLogs)
            {
                elem.Refresh();
            }
            this.@materialInfo.Refresh();
            foreach (var elem in this.@samplers)
            {
                elem.Refresh();
            }
            this.@shaderAssign.Refresh();
            foreach (var elem in this.@userDatas)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@streams)
            {
                elem.Refresh();
            }
            this.@comment.Refresh();
            this.@toolData.Refresh();
            this.@userToolData.Refresh();
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
            if (sender == this.ShaderAssign && e.PropertyName == "ShaderArchive")
            {
                this.RaisePropertyChanged("IsShaderAssigned");
            }
        }
    }
}
