﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class LinearCurve : ObservableEntity<linear_curveType>, IChildEntity, IStreamIndex, ICurve, IDeepCopyable<LinearCurve>, IDeepCopyFrom<LinearCurve>
    {
        private int @count;
        private curve_frame_typeType @frameType = curve_frame_typeType.none;
        private curve_key_typeType @keyType = curve_key_typeType.none;
        private float @scale = 1.0f;
        private float @offset = 0.0f;
        private curve_wrapType @preWrap = curve_wrapType.clamp;
        private curve_wrapType @postWrap = curve_wrapType.clamp;
        private bool @baked;
        private int @streamIndex;
        private IAnimTarget parent = null;
        private Stream @stream = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public LinearCurve()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public LinearCurve(LinearCurve source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public LinearCurve(linear_curveType source)
        {
            this.@count = source.@count;
            this.@frameType = source.@frame_type;
            this.@keyType = source.@key_type;
            this.@scale = source.@scale;
            this.@offset = source.@offset;
            this.@preWrap = source.@pre_wrap;
            this.@postWrap = source.@post_wrap;
            this.@baked = source.@baked;
            this.@streamIndex = source.@stream_index;
        }

        public int Count
        {
            get
            {
                return this.@count;
            }

            set
            {
                if (this.@count == value)
                {
                    return;
                }

                this.SetProperty(ref this.@count, value, () => this.CalcCRC());
            }
        }

        public curve_frame_typeType FrameType
        {
            get
            {
                return this.@frameType;
            }

            set
            {
                if (this.@frameType == value)
                {
                    return;
                }

                this.SetProperty(ref this.@frameType, value, () => this.CalcCRC());
            }
        }

        public curve_key_typeType KeyType
        {
            get
            {
                return this.@keyType;
            }

            set
            {
                if (this.@keyType == value)
                {
                    return;
                }

                this.SetProperty(ref this.@keyType, value, () => this.CalcCRC());
            }
        }

        public float Scale
        {
            get
            {
                return this.@scale;
            }

            set
            {
                if (this.@scale == value)
                {
                    return;
                }

                this.SetProperty(ref this.@scale, value, () => this.CalcCRC());
            }
        }

        public float Offset
        {
            get
            {
                return this.@offset;
            }

            set
            {
                if (this.@offset == value)
                {
                    return;
                }

                this.SetProperty(ref this.@offset, value, () => this.CalcCRC());
            }
        }

        public curve_wrapType PreWrap
        {
            get
            {
                return this.@preWrap;
            }

            set
            {
                if (this.@preWrap == value)
                {
                    return;
                }

                this.SetProperty(ref this.@preWrap, value, () => this.CalcCRC());
            }
        }

        public curve_wrapType PostWrap
        {
            get
            {
                return this.@postWrap;
            }

            set
            {
                if (this.@postWrap == value)
                {
                    return;
                }

                this.SetProperty(ref this.@postWrap, value, () => this.CalcCRC());
            }
        }

        public bool Baked
        {
            get
            {
                return this.@baked;
            }

            set
            {
                if (this.@baked == value)
                {
                    return;
                }

                this.SetProperty(ref this.@baked, value, () => this.CalcCRC());
            }
        }

        int IStreamIndex.StreamIndex
        {
            get
            {
                if (this.Stream != null)
                {
                    // 親が見つからない場合はインデックスを求められないので -1 を返す
                    IRootEntity rootEntity = Utility.FindRootEntity((this as IChildEntity).Parent);
                    if (rootEntity == null)
                    {
                        return -1;
                    }

                    var sourceProp = rootEntity.GetType().GetProperty("Streams").GetValue(rootEntity) as ObservableList<Stream>;
                    return sourceProp.IndexOf(this.Stream);
                }
                return this.@streamIndex;
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is IAnimTarget : true);
                this.SetProperty(ref this.parent, value as IAnimTarget, () => this.CalcCRC());
            }
        }

        public IAnimTarget Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(IAnimTarget);
            }
        }

        public Stream Stream
        {
            get
            {
                return this.@stream;
            }

            set
            {
                if (this.@stream == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@stream, value, () => this.CalcCRC());
            }
        }

        void IDeepCopyFrom<ICurve>.DeepCopyFrom(ICurve source)
        {
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.True(source is LinearCurve);
            this.DeepCopyFrom(source as LinearCurve);
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override linear_curveType CreateSerializableData()
        {
            var writeData = new linear_curveType();
            writeData.@count = this.Count;
            writeData.@frame_type = this.FrameType;
            writeData.@key_type = this.KeyType;
            writeData.@scale = this.Scale;
            writeData.@offset = this.Offset;
            writeData.@pre_wrap = this.PreWrap;
            writeData.@post_wrap = this.PostWrap;
            writeData.@baked = this.Baked;
            {
                IRootEntity rootEntity = Utility.FindRootEntity((this as IChildEntity).Parent);
                Ensure.Operation.NotNull(rootEntity);
                var sourceProp = rootEntity.GetType().GetProperty("Streams").GetValue(rootEntity) as ObservableList<Stream>;
                // 読み取り高速化のために不要なデータを読み飛ばす場合があるので、参照先が存在しない場合を許容します
                Ensure.Operation.True((sourceProp.Count == 0) || (Stream != null));
                writeData.stream_index = (this as IStreamIndex).StreamIndex;
            }
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        LinearCurve IDeepCopyable<LinearCurve>.DeepCopy()
        {
            return new LinearCurve(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(LinearCurve source)
        {
            this.@count = source.@count;
            this.@frameType = source.@frameType;
            this.@keyType = source.@keyType;
            this.@scale = source.@scale;
            this.@offset = source.@offset;
            this.@preWrap = source.@preWrap;
            this.@postWrap = source.@postWrap;
            this.@baked = source.@baked;
            this.@streamIndex = source.@streamIndex;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Count)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.FrameType)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.KeyType)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Scale)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Offset)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.PreWrap)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.PostWrap)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Baked)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32((this as IStreamIndex).StreamIndex)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
