﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class KeyShape : ObservableEntity<key_shapeType>, IDeepCopyable<KeyShape>, IDeepCopyFrom<KeyShape>
    {
        private readonly ObservableList<TargetAttrib> @targetAttribs = new ObservableList<TargetAttrib>();
        private string @name = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public KeyShape()
        {
            this.@targetAttribs.CollectionChanged += OnCollectionChanged<TargetAttrib>;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public KeyShape(KeyShape source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public KeyShape(key_shapeType source)
        {
            this.@targetAttribs.CollectionChanged += OnCollectionChanged<TargetAttrib>;
            if (source.@target_attrib_array != null)
            {
                this.@targetAttribs.Clear();
                foreach (var elem in source.@target_attrib_array.Items)
                {
                    this.@targetAttribs.Add(new TargetAttrib(elem));
                }
            }
            this.@name = source.@name;
        }

        public ObservableList<TargetAttrib> TargetAttribs
        {
            get
            {
                return this.@targetAttribs;
            }
        }

        public string Name
        {
            get
            {
                return this.@name;
            }

            set
            {
                if (this.@name == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@name, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override key_shapeType CreateSerializableData()
        {
            var writeData = new key_shapeType();
            if (this.@TargetAttribs.Count > 0)
            {
                writeData.@target_attrib_array = new target_attrib_arrayType();
                writeData.@target_attrib_array.Items = this.@targetAttribs.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@target_attrib_array.Items.Length == 0)
                {
                    writeData.@target_attrib_array = null;
                }
            }
            writeData.@name = this.Name;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        KeyShape IDeepCopyable<KeyShape>.DeepCopy()
        {
            return new KeyShape(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(KeyShape source)
        {
            CopyElements(source.@targetAttribs, this.@targetAttribs);
            this.@name = source.@name;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.TargetAttribs)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Name)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@targetAttribs)
            {
                elem.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@targetAttribs)
            {
                elem.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@targetAttribs)
            {
                elem.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
