﻿using Nintendo.ToolFoundation.Contracts;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.G3dTool.Entities.Internal
{
    internal class CopyUtility
    {
        public static object DeepCopy(object source)
        {
            Type type = source.GetType();
            if (type.IsValueType || type.Equals(typeof(string)))
            {
                // 値型か文字列
                return source;
            }
            else if (typeof(System.Collections.IList).IsAssignableFrom(type))
            {
                // 配列型
                return DeepCopyListTypeObject(source as System.Collections.IList);
            }
            else
            {
                // 参照型
                return DeepCopyReferenceTypeObject(source);
            }
        }

        private static void DeepCopy(object destination, object source)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.NotNull(destination);
            Ensure.Argument.AreEqual(destination.GetType(), source.GetType());

            var sourceProps = source.GetType().GetProperties().Where(x => x.CanRead).ToList();
            foreach (var propInfo in sourceProps)
            {
                if (!propInfo.CanRead)
                {
                    continue;
                }

                object sourceValue = propInfo.GetValue(source, null);
                if (propInfo.PropertyType.IsValueType || propInfo.PropertyType.Equals(typeof(string)))
                {
                    // 値型
                    if (propInfo.CanWrite)
                    {
                        propInfo.SetValue(destination, sourceValue, null);
                    }
                }
                else if (typeof(System.Collections.IList).IsAssignableFrom(propInfo.PropertyType))
                {
                    // 配列型
                    if (propInfo.CanWrite)
                    {
                        var instance = DeepCopyListTypeObject(sourceValue as System.Collections.IList);
                        propInfo.SetValue(destination, instance, null);
                    }
                    else
                    {
                        // 読み取り専用の場合は既に対象に値のインスタンスがあるものとする
                        var instance = propInfo.GetValue(destination, null) as System.Collections.IList;
                        DeepCopyElements(instance, sourceValue as System.Collections.IList);
                    }
                }
                else
                {
                    // 参照型
                    if (propInfo.CanWrite)
                    {
                        var instance = DeepCopyReferenceTypeObject(sourceValue);
                        propInfo.SetValue(destination, instance, null);
                    }
                }
            }
        }

        private static System.Collections.IList DeepCopyListTypeObject(System.Collections.IList source)
        {
            Ensure.Argument.NotNull(source);
            Type type = source.GetType();
            ConstructorInfo constructor = type.GetConstructor(Type.EmptyTypes);
            if (constructor == null)
            {
                // リフレクションで実装できない
                throw new Exception($"{type.FullName} does not have default consutructor");
            }

            var instance = constructor.Invoke(new object[] { }) as System.Collections.IList;
            DeepCopyElements(instance, source);

            return instance;
        }

        private static void DeepCopyElements(System.Collections.IList desitnation, System.Collections.IList source)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.NotNull(desitnation);
            desitnation.Clear();
            Type sourceType = source.GetType();
            Type itemType = GetItemType(sourceType);
            if (itemType.IsValueType)
            {
                // 値型の要素
                foreach (var elem in source)
                {
                    desitnation.Add(elem);
                }
            }
            else
            {
                // 参照型の要素
                foreach (var elem in source)
                {
                    desitnation.Add(DeepCopyReferenceTypeObject(elem));
                }
            }
        }

        private static Type GetItemType(Type collectionType)
        {
            var types = (from method in collectionType.GetMethods()
                 where method.Name == "get_Item"
                 select method.ReturnType).Distinct().ToArray();

            if (types.Length == 0)
            {
                return null;
            }

            if (types.Length != 1)
            {
                throw new Exception($"{collectionType.FullName} has multiple item types");
            }

            return types[0];
        }

        private static object DeepCopyReferenceTypeObject(object source)
        {
            Ensure.Argument.NotNull(source);
            Type type = source.GetType();
            ConstructorInfo ctor = type.GetConstructor(Type.EmptyTypes);
            if (ctor == null)
            {
                // リフレクションで実装できない
                throw new Exception($"{type.FullName} does not have default consutructor");
            }

            object instance = ctor.Invoke(new object[] { });
            DeepCopy(instance, source);
            return instance;
        }
    }
}
