﻿using Nintendo.G3dTool.Entities;
using Nintendo.G3dTool.Entities.Internal;
using nw.g3d.nw4f_3dif;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.G3dTool.Entities.Internal
{
    internal static class BoneExtensions
    {
        internal static Float4x3 CalculateInverseTransformMatrix(this Bone targetBone)
        {
            Float4x3 invTransformMat43 = new Float4x3();
            Float4x4 invTransformMat44 = targetBone.CalculateTransformMatrix().Inverse();
            if (invTransformMat44 == null)
            {
                return null;
            }

            for (int row = 0; row < 3; ++row)
            {
                for (int col = 0; col < 4; ++col)
                {
                    invTransformMat43[row, col] = invTransformMat44[row, col];
                }
            }

            return invTransformMat43;
        }

        internal static Float4x4 CalculateTransformStandard(this Bone targetBone)
        {
            // ルートノードから boneInfo までの変換を合成する。
            Float4x4 transformMatrix = Float4x4.Identity;
            foreach (Bone bone in targetBone.FindBonePathToRootBone())
            {
                Float4x4 rotateMatrix = bone.CalculateRotationMatrix();
                Float4x4 translateMatrix = bone.CalculateTranslationMatrix();
                Float4x4 scaleMatrix = bone.CalculateScaleMatrix();
                transformMatrix *= translateMatrix * rotateMatrix * scaleMatrix;
            }

            return transformMatrix;
        }

        internal static Float4x4 CalculateTransformMaya(this Bone targetBone)
        {
            // ルートノードから Bone までの変換を合成する。
            Float4x4 translateRotateMatrix = Float4x4.Identity;
            Float4x4 parentScaleMatrix = new Float4x4(); // 親のスケール行列
            Float3 parentScale = new Float3(1.0f, 1.0f, 1.0f);
            foreach (Bone bone in targetBone.FindBonePathToRootBone())
            {
                Float4x4 rotateMatrix = bone.CalculateRotationMatrix();
                Float4x4 translateMatrix = bone.CalculateTranslationMatrix();

                // scale_compensate の設定に従って変換を求める。
                if (bone.ScaleCompensate)
                {
                    // scale_compensate が ON の場合は親のスケールを translate にだけ適用する
                    translateMatrix[0, 3] *= parentScale.X;
                    translateMatrix[1, 3] *= parentScale.Y;
                    translateMatrix[2, 3] *= parentScale.Z;
                    translateRotateMatrix *= translateMatrix * rotateMatrix;
                }
                else
                {
                    parentScaleMatrix.SetScale(parentScale);
                    translateRotateMatrix *= parentScaleMatrix * translateMatrix * rotateMatrix;
                }

                parentScale = bone.Scale;
            }

            // この時点で parentScale には bone.Scale が設定されている。
            // 最後に bone のスケールを適用する。
            Float4x4 scaleMatrix = new Float4x4();
            scaleMatrix.SetScale(parentScale);
            return translateRotateMatrix * scaleMatrix;
        }

        private static List<Bone> FindBonePathToRootBone(this Bone targetBone)
        {
            // Bone からルートノードまでのボーンリストを作成する
            // ボーンリストの先頭には引数で渡されたボーンを登録しておく。
            List<Bone> bonePath = new List<Bone>();
            bonePath.Add(targetBone);
            Bone parent = targetBone.ParentBone;
            int boneSize = targetBone.Parent.CountBones();
            int count = 0;
            for (; count < boneSize && parent != null; ++count)
            {
                bonePath.Add(parent);
                parent = parent.ParentBone;
            }

            // 親をたどっていった数とモデル中のボーン数が一致すると
            // 参照が循環しているということなのでエラーとする。
            if (count == boneSize)
            {
                throw new Exception($"<bone> \"{targetBone.Name}\" has cyclic indexing.");
            }

            // 変換はルートノードから順番に計算していくので、ボーンリストの順番を反転させる。
            bonePath.Reverse();
            return bonePath;
        }

        private static Float4x4 CalculateRotationMatrix(this Bone targetBone)
        {
            Float4x4 rotateMatrix = new Float4x4();
            switch (targetBone.Parent.SkeletonInfo.RotateMode)
            {
                case skeleton_info_rotate_modeType.euler_xyz:
                    {
                        Float3 euler = new Float3(
                            (targetBone.Rotate.X / 180.0f) * (float)Math.PI,
                            (targetBone.Rotate.Y / 180.0f) * (float)Math.PI,
                            (targetBone.Rotate.Z / 180.0f) * (float)Math.PI);
                        rotateMatrix.SetRotationEulerXyz(euler.X, euler.Y, euler.Z);
                    }
                    break;
                case skeleton_info_rotate_modeType.quaternion:
                    {
                        rotateMatrix.SetRotationQuaternion(targetBone.Rotate.X, targetBone.Rotate.Y, targetBone.Rotate.Z, targetBone.Rotate.W);
                    }
                    break;
                default:
                    throw new Exception($"Unexpected scale mode{targetBone.Parent.SkeletonInfo.RotateMode}");
            }

            return rotateMatrix;
        }

        private static Float4x4 CalculateTranslationMatrix(this Bone targetBone)
        {
            Float4x4 translateMatrix = new Float4x4();
            translateMatrix.SetTranslation(targetBone.Translate);
            return translateMatrix;
        }

        private static Float4x4 CalculateScaleMatrix(this Bone targetBone)
        {
            Float4x4 scaleMatrix = new Float4x4();
            scaleMatrix.SetScale(targetBone.Scale);
            return scaleMatrix;
        }
    }
}
