﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;
namespace Nintendo.G3dTool.Entities{
    public class Float4x4 : ObservableEntity<string>, IMatrix, IEquatable<Float4x4>
    {
        private readonly float[] values = new float[16];

        public Float4x4()
        {
            this.SetToIdentity();
        }

        public Float4x4(
            float elem00,
            float elem01,
            float elem02,
            float elem03,
            float elem10,
            float elem11,
            float elem12,
            float elem13,
            float elem20,
            float elem21,
            float elem22,
            float elem23,
            float elem30,
            float elem31,
            float elem32,
            float elem33)
            : this()
        {
            this[0] = elem00;
            this[1] = elem01;
            this[2] = elem02;
            this[3] = elem03;
            this[4] = elem10;
            this[5] = elem11;
            this[6] = elem12;
            this[7] = elem13;
            this[8] = elem20;
            this[9] = elem21;
            this[10] = elem22;
            this[11] = elem23;
            this[12] = elem30;
            this[13] = elem31;
            this[14] = elem32;
            this[15] = elem33;
        }

        internal float[] Values
        {
            get
            {
                return this.values;
            }
        }

        public float this[int index]
        {
            get
            {
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(index, 0, 16);
                return this.values[index];
            }

            set
            {
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(index, 0, 16);
                this.SetProperty(ref this.values[index], value, () => this.CalcCRC());
            }
        }


        private float Get(int rowIndex, int columnIndex)
        {
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(rowIndex, 0, 4);
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(columnIndex, 0, 4);

            return this[rowIndex * 4 + columnIndex];
        }

        public int RowCount
        {
            get
            {
                return 4;
            }
        }

        public int ColumnCount
        {
            get
            {
                return 4;
            }
        }

        public static Float4x4 Identity
        {
            get
            {
                return new Float4x4();
            }
        }

        public float this[int rowIndex, int columnIndex]
        {
            get
            {
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(rowIndex, 0, 4);
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(columnIndex, 0, 4);
                return this.values[rowIndex * 4 + columnIndex];
            }

            set
            {
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(rowIndex, 0, 4);
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(columnIndex, 0, 4);
                this.SetProperty(ref this.values[rowIndex * 4 + columnIndex], value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 行成分のベクトル列を取得します。
        /// </summary>
        public IVector[] Rows
        {
            get
            {
                IVector[] data =
                {
                    new Float4(this[0, 0], this[0, 1], this[0, 2], this[0, 3]),
                    new Float4(this[1, 0], this[1, 1], this[1, 2], this[1, 3]),
                    new Float4(this[2, 0], this[2, 1], this[2, 2], this[2, 3]),
                    new Float4(this[3, 0], this[3, 1], this[3, 2], this[3, 3]),
                };
                return data;
            }
        }

        /// <summary>
        /// 列成分のベクトル列を取得します。
        /// </summary>
        public IVector[] Columns
        {
            get
            {
                IVector[] data =
                {
                    new Float4(this[0, 0], this[1, 0], this[2, 0], this[3, 0]),
                    new Float4(this[0, 1], this[1, 1], this[2, 1], this[3, 1]),
                    new Float4(this[0, 2], this[1, 2], this[2, 2], this[3, 2]),
                    new Float4(this[0, 3], this[1, 3], this[2, 3], this[3, 3]),
                };

                return data;
            }
        }

        public void SetToIdentity()
        {
            Array.ForEach(this.values, x => x = 0.0f);
            this[0, 0] = 1.0f;
            this[1, 1] = 1.0f;
            this[2, 2] = 1.0f;
            this[3, 3] = 1.0f;
        }

        /// <summary>
        /// 行列とスカラーを乗算します。
        /// </summary>
        /// <param name="mtx">演算の左の行列です。</param>
        /// <param name="scalar">演算の右のスカラーです。</param>
        /// <returns>演算結果です。</returns>
        public static Float4x4 operator *(Float4x4 mtx, float scalar)
        {
            var result = new Float4x4();
            for (int i = 0; i < mtx.RowCount; ++i)
            {
                for (int j = 0; j < mtx.ColumnCount; ++j)
                {
                    result[i, j] = mtx[i, j] * scalar;
                }
            }

            return result;
        }


        /// <summary>
        /// 2 つの行列を乗算します。
        /// </summary>
        /// <param name="rhs">演算の右の行列です。</param>
        /// <returns>演算結果です。</returns>
        public static Float4x4 operator *(Float4x4 lhs, Float4x4 rhs)
        {
            int dimension = lhs.ColumnCount;
            var result = new Float4x4();
            for (int i = 0; i < lhs.RowCount; ++i)
            {
                for (int j = 0; j < lhs.ColumnCount; ++j)
                {
                    float sum = 0f;
                    for (int k = 0; k < dimension; ++k)
                    {
                        sum += lhs[i, k] * rhs[k, j];
                    }

                    result[i, j] = sum;
                }
            }

            return result;
        }

        /// <summary>
        /// 行列にベクトルを乗算します。
        /// </summary>
        /// <param name="mtx">演算の左の行列です。</param>
        /// <param name="vec">演算の右のベクトルです。</param>
        /// <returns>演算結果です。</returns>
        public static Float3 operator *(Float4x4 mtx, Float3 vec)
        {
            Float3 result = new Float3();
            result.X = (mtx[0, 0] * vec.X) + (mtx[0, 1] * vec.Y) + (mtx[0, 2] * vec.Z) + mtx[0, 3];
            result.Y = (mtx[1, 0] * vec.X) + (mtx[1, 1] * vec.Y) + (mtx[1, 2] * vec.Z) + mtx[1, 3];
            result.Z = (mtx[2, 0] * vec.X) + (mtx[2, 1] * vec.Y) + (mtx[2, 2] * vec.Z) + mtx[2, 3];
            return result;
        }

        /// <summary>
        /// 逆行列に変換します。
        /// </summary>
        public void Invert()
        {
            Float4x4 inverse = this.Inverse();
            for (int row = 0; row < 4; ++row)
            {
                for (int col = 0; col < 4; ++col)
                {
                    this[row, col] = inverse[row, col];
                }
            }
        }

        public void SetToZero()
        {
            for (int elemIndex = 0; elemIndex < 16; ++elemIndex)
            {
                this[elemIndex] = 0;
            }
        }

        /// <summary>
        /// 比較演算子
        /// </summary>
        /// <param name="obj"></param>
        /// <returns>一致していれば true、そうでなければ false を返します。</returns>
        public bool Equals(Float4x4 obj)
        {
            if (obj == null)
            {
                return false;
            }

            bool isEqual = true;
            for (int elemIndex = 0; elemIndex < 16; ++elemIndex)
            {
                isEqual &= (obj[elemIndex] == this[elemIndex]);
            }

            return isEqual;
        }

        public override string CreateSerializableData()
        {
            return this.ToString();
        }

        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            return crc.ComputeHashUInt32(this.ToString());
        }

        public override string ToString()
        {
            string result = string.Empty;
            for (int elemIndex = 0; elemIndex < 16; ++elemIndex)
            {
                result += Convert.ToDouble(this[elemIndex]).ToString() + " ";
            }

            return result.TrimEnd();
        }

        public void DeepCopyFrom(Float4x4 source)
        {
            for (int elemIndex = 0; elemIndex < 16; ++elemIndex)
            {
                this[elemIndex] = source.Values[elemIndex];
            }
        }

        public static Float4x4 Parse(string text)
        {
            string[] valueTexts = StringUtility.SplitValueListText(text);
            Float4x4 result = new Float4x4();
            for (int elemIndex = 0; elemIndex < 16; ++elemIndex)
            {
                result[elemIndex] = float.Parse(valueTexts[elemIndex]);
            }

            return result;
        }
    }
}
