﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;
namespace Nintendo.G3dTool.Entities{
    public class Float3x4 : ObservableEntity<string>, IMatrix, IEquatable<Float3x4>
    {
        private readonly float[] values = new float[12];

        public Float3x4()
        {
            this.SetToIdentity();
        }

        public Float3x4(
            float elem00,
            float elem01,
            float elem02,
            float elem10,
            float elem11,
            float elem12,
            float elem20,
            float elem21,
            float elem22,
            float elem30,
            float elem31,
            float elem32)
            : this()
        {
            this[0] = elem00;
            this[1] = elem01;
            this[2] = elem02;
            this[3] = elem10;
            this[4] = elem11;
            this[5] = elem12;
            this[6] = elem20;
            this[7] = elem21;
            this[8] = elem22;
            this[9] = elem30;
            this[10] = elem31;
            this[11] = elem32;
        }

        internal float[] Values
        {
            get
            {
                return this.values;
            }
        }

        public float this[int index]
        {
            get
            {
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(index, 0, 12);
                return this.values[index];
            }

            set
            {
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(index, 0, 12);
                this.SetProperty(ref this.values[index], value, () => this.CalcCRC());
            }
        }


        private float Get(int rowIndex, int columnIndex)
        {
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(rowIndex, 0, 4);
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(columnIndex, 0, 3);

            return this[rowIndex * 3 + columnIndex];
        }

        public int RowCount
        {
            get
            {
                return 4;
            }
        }

        public int ColumnCount
        {
            get
            {
                return 3;
            }
        }

        public static Float3x4 Identity
        {
            get
            {
                return new Float3x4();
            }
        }

        public float this[int rowIndex, int columnIndex]
        {
            get
            {
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(rowIndex, 0, 4);
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(columnIndex, 0, 3);
                return this.values[rowIndex * 3 + columnIndex];
            }

            set
            {
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(rowIndex, 0, 4);
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(columnIndex, 0, 3);
                this.SetProperty(ref this.values[rowIndex * 3 + columnIndex], value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 行成分のベクトル列を取得します。
        /// </summary>
        public IVector[] Rows
        {
            get
            {
                IVector[] data =
                {
                    new Float3(this[0, 0], this[0, 1], this[0, 2]),
                    new Float3(this[1, 0], this[1, 1], this[1, 2]),
                    new Float3(this[2, 0], this[2, 1], this[2, 2]),
                    new Float3(this[3, 0], this[3, 1], this[3, 2]),
                };
                return data;
            }
        }

        /// <summary>
        /// 列成分のベクトル列を取得します。
        /// </summary>
        public IVector[] Columns
        {
            get
            {
                IVector[] data =
                {
                    new Float4(this[0, 0], this[1, 0], this[2, 0], this[3, 0]),
                    new Float4(this[0, 1], this[1, 1], this[2, 1], this[3, 1]),
                    new Float4(this[0, 2], this[1, 2], this[2, 2], this[3, 2]),
                };

                return data;
            }
        }

        public void SetToIdentity()
        {
            Array.ForEach(this.values, x => x = 0.0f);
            this[0, 0] = 1.0f;
            this[1, 1] = 1.0f;
            this[2, 2] = 1.0f;
        }

        /// <summary>
        /// 行列とスカラーを乗算します。
        /// </summary>
        /// <param name="mtx">演算の左の行列です。</param>
        /// <param name="scalar">演算の右のスカラーです。</param>
        /// <returns>演算結果です。</returns>
        public static Float3x4 operator *(Float3x4 mtx, float scalar)
        {
            var result = new Float3x4();
            for (int i = 0; i < mtx.RowCount; ++i)
            {
                for (int j = 0; j < mtx.ColumnCount; ++j)
                {
                    result[i, j] = mtx[i, j] * scalar;
                }
            }

            return result;
        }


        public void SetToZero()
        {
            for (int elemIndex = 0; elemIndex < 12; ++elemIndex)
            {
                this[elemIndex] = 0;
            }
        }

        /// <summary>
        /// 比較演算子
        /// </summary>
        /// <param name="obj"></param>
        /// <returns>一致していれば true、そうでなければ false を返します。</returns>
        public bool Equals(Float3x4 obj)
        {
            if (obj == null)
            {
                return false;
            }

            bool isEqual = true;
            for (int elemIndex = 0; elemIndex < 12; ++elemIndex)
            {
                isEqual &= (obj[elemIndex] == this[elemIndex]);
            }

            return isEqual;
        }

        public override string CreateSerializableData()
        {
            return this.ToString();
        }

        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            return crc.ComputeHashUInt32(this.ToString());
        }

        public override string ToString()
        {
            string result = string.Empty;
            for (int elemIndex = 0; elemIndex < 12; ++elemIndex)
            {
                result += Convert.ToDouble(this[elemIndex]).ToString() + " ";
            }

            return result.TrimEnd();
        }

        public void DeepCopyFrom(Float3x4 source)
        {
            for (int elemIndex = 0; elemIndex < 12; ++elemIndex)
            {
                this[elemIndex] = source.Values[elemIndex];
            }
        }

        public static Float3x4 Parse(string text)
        {
            string[] valueTexts = StringUtility.SplitValueListText(text);
            Float3x4 result = new Float3x4();
            for (int elemIndex = 0; elemIndex < 12; ++elemIndex)
            {
                result[elemIndex] = float.Parse(valueTexts[elemIndex]);
            }

            return result;
        }
    }
}
