﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Comment : ObservableEntity<commentType>, IDeepCopyable<Comment>, IDeepCopyFrom<Comment>
    {
        private string @label = string.Empty;
        private RgbColor @color = new RgbColor(0.0, 0.0, 0.0);
        private string @text = string.Empty;
        private bool @isColorEnabled = false;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Comment()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Comment(Comment source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Comment(commentType source)
        {
            this.@label = source.@label;
            if (!string.IsNullOrEmpty(source.color))
            {
                this.IsColorEnabled = true;
                float[] colorElems = StringUtility.ConvertTextToValueArray<float>(source.color);
                this.@color.R = colorElems[0];
                this.@color.G = colorElems[1];
                this.@color.B = colorElems[2];
            }
            this.@text = source.@text;
        }

        public string Label
        {
            get
            {
                return this.@label;
            }

            set
            {
                if (this.@label == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@label, value, () => this.CalcCRC());
            }
        }

        public RgbColor Color
        {
            get
            {
                return this.@color;
            }

            set
            {
                if (this.@color == value)
                {
                    return;
                }

                this.SetProperty(ref this.@color, value, () => this.CalcCRC());
            }
        }

        public string Text
        {
            get
            {
                return this.@text;
            }

            set
            {
                if (this.@text == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@text, value, () => this.CalcCRC());
            }
        }

        public bool IsColorEnabled
        {
            get
            {
                return this.@isColorEnabled;
            }

            set
            {
                if (this.@isColorEnabled == value)
                {
                    return;
                }

                this.SetProperty(ref this.@isColorEnabled, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override commentType CreateSerializableData()
        {
            if (!this.IsColorEnabled && string.IsNullOrEmpty(this.label) && string.IsNullOrEmpty(this.text))
            {
                return null;
            }
            var writeData = new commentType();
            writeData.@label = this.Label;
            if (this.IsColorEnabled)
            {
                writeData.color = this.Color.ToString();
            }
            else
            {
                writeData.color = string.Empty;
            }
            writeData.@text = this.Text;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Comment IDeepCopyable<Comment>.DeepCopy()
        {
            return new Comment(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Comment source)
        {
            this.@label = source.@label;
            this.@color = source.@color;
            this.@text = source.@text;
            this.@isColorEnabled = source.@isColorEnabled;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Label)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Color)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Text)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.IsColorEnabled)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
